"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3SourceAction = exports.S3Trigger = void 0;
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const targets = require("../../../aws-events-targets"); // Automatically re-written from '@aws-cdk/aws-events-targets'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * How should the S3 Action detect changes.
 * This is the type of the {@link S3SourceAction.trigger} property.
 */
var S3Trigger;
(function (S3Trigger) {
    /**
     * The Action will never detect changes -
     * the Pipeline it's part of will only begin a run when explicitly started.
     */
    S3Trigger["NONE"] = "None";
    /**
     * CodePipeline will poll S3 to detect changes.
     * This is the default method of detecting changes.
     */
    S3Trigger["POLL"] = "Poll";
    /**
     * CodePipeline will use CloudWatch Events to be notified of changes.
     * Note that the Bucket that the Action uses needs to be part of a CloudTrail Trail
     * for the events to be delivered.
     */
    S3Trigger["EVENTS"] = "Events";
})(S3Trigger = exports.S3Trigger || (exports.S3Trigger = {}));
/**
 * Source that is provided by a specific Amazon S3 object.
 *
 * Will trigger the pipeline as soon as the S3 object changes, but only if there is
 * a CloudTrail Trail in the account that captures the S3 event.
 */
class S3SourceAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            resource: props.bucket,
            category: codepipeline.ActionCategory.SOURCE,
            provider: 'S3',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        if (props.bucketKey.length === 0) {
            throw new Error('Property bucketKey cannot be an empty string');
        }
        this.props = props;
    }
    /** The variables emitted by this action. */
    get variables() {
        return {
            versionId: this.variableExpression('VersionId'),
            eTag: this.variableExpression('ETag'),
        };
    }
    bound(_scope, stage, options) {
        if (this.props.trigger === S3Trigger.EVENTS) {
            const id = this.generateEventId(stage);
            this.props.bucket.onCloudTrailWriteObject(id, {
                target: new targets.CodePipeline(stage.pipeline),
                paths: [this.props.bucketKey],
            });
        }
        // we need to read from the source bucket...
        this.props.bucket.grantRead(options.role);
        // ...and write to the Pipeline bucket
        options.bucket.grantWrite(options.role);
        return {
            configuration: {
                S3Bucket: this.props.bucket.bucketName,
                S3ObjectKey: this.props.bucketKey,
                PollForSourceChanges: this.props.trigger && this.props.trigger === S3Trigger.POLL,
            },
        };
    }
    generateEventId(stage) {
        let ret;
        const baseId = stage.pipeline.node.uniqueId + 'SourceEventRule';
        if (core_1.Token.isUnresolved(this.props.bucketKey)) {
            // If bucketKey is a Token, don't include it in the ID.
            // Instead, use numbers to differentiate if multiple actions observe the same bucket
            let candidate = baseId;
            let counter = 0;
            while (this.props.bucket.node.tryFindChild(candidate) !== undefined) {
                counter += 1;
                candidate = baseId + counter;
            }
            ret = candidate;
        }
        else {
            // we can't use Tokens in construct IDs,
            // however, if bucketKey is not a Token,
            // we want it to differentiate between multiple actions
            // observing the same Bucket with different keys
            ret = baseId + this.props.bucketKey;
            if (this.props.bucket.node.tryFindChild(ret)) {
                // this means a duplicate path for the same bucket - error out
                throw new Error(`S3 source action with path '${this.props.bucketKey}' is already present in the pipeline for this source bucket`);
            }
        }
        return ret;
    }
}
exports.S3SourceAction = S3SourceAction;
//# sourceMappingURL=data:application/json;base64,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