"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StepFunctionInvokeAction = exports.StateMachineInput = void 0;
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const action_1 = require("../action");
/**
 * Represents the input for the StateMachine.
 */
class StateMachineInput {
    constructor(input, inputArtifact, inputType) {
        this.input = input;
        this.inputArtifact = inputArtifact;
        this.inputType = inputType;
    }
    /**
     * When the input type is FilePath, input artifact and
     * filepath must be specified.
     */
    static filePath(inputFile) {
        return new StateMachineInput(inputFile.location, inputFile.artifact, 'FilePath');
    }
    /**
     * When the input type is Literal, input value is passed
     * directly to the state machine input.
     */
    static literal(object) {
        return new StateMachineInput(JSON.stringify(object), undefined, 'Literal');
    }
}
exports.StateMachineInput = StateMachineInput;
/**
 * StepFunctionInvokeAction that is provided by an AWS CodePipeline.
 */
class StepFunctionInvokeAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            resource: props.stateMachine,
            category: codepipeline.ActionCategory.INVOKE,
            provider: 'StepFunctions',
            artifactBounds: {
                minInputs: 0,
                maxInputs: 1,
                minOutputs: 0,
                maxOutputs: 1,
            },
            inputs: (props.stateMachineInput && props.stateMachineInput.inputArtifact) ? [props.stateMachineInput.inputArtifact] : [],
            outputs: (props.output) ? [props.output] : [],
        });
        this.props = props;
    }
    bound(_scope, _stage, options) {
        var _a, _b, _c, _d, _e;
        // allow pipeline to invoke this step function
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['states:StartExecution', 'states:DescribeStateMachine'],
            resources: [this.props.stateMachine.stateMachineArn],
        }));
        // allow state machine executions to be inspected
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['states:DescribeExecution'],
            resources: [cdk.Stack.of(this.props.stateMachine).formatArn({
                    service: 'states',
                    resource: 'execution',
                    resourceName: `${this.props.stateMachine.stateMachineArn}:${(_a = this.props.executionNamePrefix) !== null && _a !== void 0 ? _a : ''}*`,
                    sep: ':',
                })],
        }));
        // allow the Role access to the Bucket, if there are any inputs/outputs
        if (((_b = this.actionProperties.inputs) !== null && _b !== void 0 ? _b : []).length > 0) {
            options.bucket.grantRead(options.role);
        }
        if (((_c = this.actionProperties.outputs) !== null && _c !== void 0 ? _c : []).length > 0) {
            options.bucket.grantWrite(options.role);
        }
        return {
            configuration: {
                StateMachineArn: this.props.stateMachine.stateMachineArn,
                Input: (_d = this.props.stateMachineInput) === null || _d === void 0 ? void 0 : _d.input,
                InputType: (_e = this.props.stateMachineInput) === null || _e === void 0 ? void 0 : _e.inputType,
                ExecutionNamePrefix: this.props.executionNamePrefix,
            },
        };
    }
}
exports.StepFunctionInvokeAction = StepFunctionInvokeAction;
//# sourceMappingURL=data:application/json;base64,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