"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPoolDomain = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const custom_resources_1 = require("../../custom-resources"); // Automatically re-written from '@aws-cdk/custom-resources'
const cognito_generated_1 = require("./cognito.generated");
/**
 * Define a user pool domain
 */
class UserPoolDomain extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c, _d;
        super(scope, id);
        if (!!props.customDomain === !!props.cognitoDomain) {
            throw new Error('One of, and only one of, cognitoDomain or customDomain must be specified');
        }
        if (((_a = props.cognitoDomain) === null || _a === void 0 ? void 0 : _a.domainPrefix) &&
            !core_1.Token.isUnresolved((_b = props.cognitoDomain) === null || _b === void 0 ? void 0 : _b.domainPrefix) &&
            !/^[a-z0-9-]+$/.test(props.cognitoDomain.domainPrefix)) {
            throw new Error('domainPrefix for cognitoDomain can contain only lowercase alphabets, numbers and hyphens');
        }
        this.isCognitoDomain = !!props.cognitoDomain;
        const domainName = ((_c = props.cognitoDomain) === null || _c === void 0 ? void 0 : _c.domainPrefix) || ((_d = props.customDomain) === null || _d === void 0 ? void 0 : _d.domainName);
        const resource = new cognito_generated_1.CfnUserPoolDomain(this, 'Resource', {
            userPoolId: props.userPool.userPoolId,
            domain: domainName,
            customDomainConfig: props.customDomain ? { certificateArn: props.customDomain.certificate.certificateArn } : undefined,
        });
        this.domainName = resource.ref;
    }
    /**
     * Import a UserPoolDomain given its domain name
     */
    static fromDomainName(scope, id, userPoolDomainName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.domainName = userPoolDomainName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * The domain name of the CloudFront distribution associated with the user pool domain.
     */
    get cloudFrontDomainName() {
        if (!this.cloudFrontCustomResource) {
            const sdkCall = {
                service: 'CognitoIdentityServiceProvider',
                action: 'describeUserPoolDomain',
                parameters: {
                    Domain: this.domainName,
                },
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(this.domainName),
            };
            this.cloudFrontCustomResource = new custom_resources_1.AwsCustomResource(this, 'CloudFrontDomainName', {
                resourceType: 'Custom::UserPoolCloudFrontDomainName',
                onCreate: sdkCall,
                onUpdate: sdkCall,
                policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                    // DescribeUserPoolDomain only supports access level '*'
                    // https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncognitouserpools.html#amazoncognitouserpools-actions-as-permissions
                    resources: ['*'],
                }),
            });
        }
        return this.cloudFrontCustomResource.getResponseField('DomainDescription.CloudFrontDistribution');
    }
    /**
     * The URL to the hosted UI associated with this domain
     */
    baseUrl() {
        if (this.isCognitoDomain) {
            return `https://${this.domainName}.auth.${core_1.Stack.of(this).region}.amazoncognito.com`;
        }
        return `https://${this.domainName}`;
    }
    /**
     * The URL to the sign in page in this domain using a specific UserPoolClient
     * @param client [disable-awslint:ref-via-interface] the user pool client that the UI will use to interact with the UserPool
     * @param options options to customize the behaviour of this method.
     */
    signInUrl(client, options) {
        var _a;
        let responseType;
        if (client.oAuthFlows.authorizationCodeGrant) {
            responseType = 'code';
        }
        else if (client.oAuthFlows.implicitCodeGrant) {
            responseType = 'token';
        }
        else {
            throw new Error('signInUrl is not supported for clients without authorizationCodeGrant or implicitCodeGrant flow enabled');
        }
        const path = (_a = options.signInPath) !== null && _a !== void 0 ? _a : '/login';
        return `${this.baseUrl()}${path}?client_id=${client.userPoolClientId}&response_type=${responseType}&redirect_uri=${options.redirectUri}`;
    }
}
exports.UserPoolDomain = UserPoolDomain;
//# sourceMappingURL=data:application/json;base64,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