"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFormationStackNotificationCheck = exports.CloudFormationStackDriftDetectionCheck = exports.AccessKeysRotated = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const rule_1 = require("./rule");
/**
 * Checks whether the active access keys are rotated within the number of days
 * specified in `maxAge`.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/access-keys-rotated.html
 *
 * @resource AWS::Config::ConfigRule
 */
class AccessKeysRotated extends rule_1.ManagedRule {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            ...props,
            identifier: 'ACCESS_KEYS_ROTATED',
            inputParameters: {
                ...props.maxAge
                    ? {
                        maxAccessKeyAge: props.maxAge.toDays(),
                    }
                    : {},
            },
        });
    }
}
exports.AccessKeysRotated = AccessKeysRotated;
/**
 * Checks whether your CloudFormation stacks' actual configuration differs, or
 * has drifted, from its expected configuration.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudformation-stack-drift-detection-check.html
 *
 * @resource AWS::Config::ConfigRule
 */
class CloudFormationStackDriftDetectionCheck extends rule_1.ManagedRule {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            ...props,
            identifier: 'CLOUDFORMATION_STACK_DRIFT_DETECTION_CHECK',
            inputParameters: {
                cloudformationRoleArn: core_1.Lazy.stringValue({ produce: () => this.role.roleArn }),
            },
        });
        this.scopeToResource('AWS::CloudFormation::Stack', props.ownStackOnly ? core_1.Stack.of(this).stackId : undefined);
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('config.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('ReadOnlyAccess'),
            ],
        });
    }
}
exports.CloudFormationStackDriftDetectionCheck = CloudFormationStackDriftDetectionCheck;
/**
 * Checks whether your CloudFormation stacks are sending event notifications to
 * a SNS topic. Optionally checks whether specified SNS topics are used.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudformation-stack-notification-check.html
 *
 * @resource AWS::Config::ConfigRule
 */
class CloudFormationStackNotificationCheck extends rule_1.ManagedRule {
    constructor(scope, id, props = {}) {
        if (props.topics && props.topics.length > 5) {
            throw new Error('At most 5 topics can be specified.');
        }
        super(scope, id, {
            ...props,
            identifier: 'CLOUDFORMATION_STACK_NOTIFICATION_CHECK',
            inputParameters: props.topics && props.topics.reduce((params, topic, idx) => ({ ...params, [`snsTopic${idx + 1}`]: topic.topicArn }), {}),
        });
        this.scopeToResource('AWS::CloudFormation::Stack');
    }
}
exports.CloudFormationStackNotificationCheck = CloudFormationStackNotificationCheck;
//# sourceMappingURL=data:application/json;base64,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