"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseCluster = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const database_secret_1 = require("./database-secret");
const docdb_generated_1 = require("./docdb.generated");
const endpoint_1 = require("./endpoint");
/**
 * A new or imported clustered database.
 */
class DatabaseClusterBase extends core_1.Resource {
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.clusterIdentifier,
            targetType: secretsmanager.AttachmentTargetType.DOCDB_DB_CLUSTER,
        };
    }
}
/**
 * Create a clustered database with a given number of instances.
 *
 * @resource AWS::DocDB::DBCluster
 */
class DatabaseCluster extends DatabaseClusterBase {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g;
        super(scope, id);
        /**
         * Identifiers of the replicas
         */
        this.instanceIdentifiers = [];
        /**
         * Endpoints which address each individual replica.
         */
        this.instanceEndpoints = [];
        this.vpc = props.instanceProps.vpc;
        this.vpcSubnets = props.instanceProps.vpcSubnets;
        // Determine the subnet(s) to deploy the DocDB cluster to
        const { subnetIds, internetConnectivityEstablished } = this.vpc.selectSubnets(this.vpcSubnets);
        // DocDB clusters require a subnet group with subnets from at least two AZs.
        // We cannot test whether the subnets are in different AZs, but at least we can test the amount.
        // See https://docs.aws.amazon.com/documentdb/latest/developerguide/replication.html#replication.high-availability
        if (subnetIds.length < 2) {
            throw new Error(`Cluster requires at least 2 subnets, got ${subnetIds.length}`);
        }
        const subnetGroup = new docdb_generated_1.CfnDBSubnetGroup(this, 'Subnets', {
            dbSubnetGroupDescription: `Subnets for ${id} database`,
            subnetIds,
        });
        // Create the security group for the DB cluster
        let securityGroup;
        if (props.instanceProps.securityGroup) {
            securityGroup = props.instanceProps.securityGroup;
        }
        else {
            securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: 'DocumentDB security group',
                vpc: this.vpc,
            });
            // HACK: Use an escape-hatch to apply a consistent removal policy to the
            // security group so we don't get errors when trying to delete the stack
            securityGroup.node.defaultChild.applyRemovalPolicy(props.removalPolicy, {
                applyToUpdateReplacePolicy: true,
            });
        }
        this.securityGroupId = securityGroup.securityGroupId;
        // Create the secret manager secret if no password is specified
        let secret;
        if (!props.masterUser.password) {
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: props.masterUser.username,
                encryptionKey: props.masterUser.kmsKey,
            });
        }
        // Default to encrypted storage
        const storageEncrypted = (_a = props.storageEncrypted) !== null && _a !== void 0 ? _a : true;
        if (props.kmsKey && !storageEncrypted) {
            throw new Error('KMS key supplied but storageEncrypted is false');
        }
        // Create the DocDB cluster
        const cluster = new docdb_generated_1.CfnDBCluster(this, 'Resource', {
            // Basic
            engineVersion: props.engineVersion,
            dbClusterIdentifier: props.dbClusterName,
            dbSubnetGroupName: subnetGroup.ref,
            port: props.port,
            vpcSecurityGroupIds: [this.securityGroupId],
            dbClusterParameterGroupName: (_b = props.parameterGroup) === null || _b === void 0 ? void 0 : _b.parameterGroupName,
            // Admin
            masterUsername: secret ? secret.secretValueFromJson('username').toString() : props.masterUser.username,
            masterUserPassword: secret
                ? secret.secretValueFromJson('password').toString()
                : props.masterUser.password.toString(),
            // Backup
            backupRetentionPeriod: (_d = (_c = props.backup) === null || _c === void 0 ? void 0 : _c.retention) === null || _d === void 0 ? void 0 : _d.toDays(),
            preferredBackupWindow: (_e = props.backup) === null || _e === void 0 ? void 0 : _e.preferredWindow,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            // Encryption
            kmsKeyId: (_f = props.kmsKey) === null || _f === void 0 ? void 0 : _f.keyArn,
            storageEncrypted,
        });
        cluster.applyRemovalPolicy(props.removalPolicy, {
            applyToUpdateReplacePolicy: true,
        });
        this.clusterIdentifier = cluster.ref;
        this.clusterResourceIdentifier = cluster.attrClusterResourceId;
        const port = core_1.Token.asNumber(cluster.attrPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpoint, port);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(cluster.attrReadEndpoint, port);
        if (secret) {
            this.secret = secret.attach(this);
        }
        // Create the instances
        const instanceCount = (_g = props.instances) !== null && _g !== void 0 ? _g : DatabaseCluster.DEFAULT_NUM_INSTANCES;
        if (instanceCount < 1) {
            throw new Error('At least one instance is required');
        }
        for (let i = 0; i < instanceCount; i++) {
            const instanceIndex = i + 1;
            const instanceIdentifier = props.instanceIdentifierBase != null ? `${props.instanceIdentifierBase}${instanceIndex}`
                : props.dbClusterName != null ? `${props.dbClusterName}instance${instanceIndex}` : undefined;
            const instance = new docdb_generated_1.CfnDBInstance(this, `Instance${instanceIndex}`, {
                // Link to cluster
                dbClusterIdentifier: cluster.ref,
                dbInstanceIdentifier: instanceIdentifier,
                // Instance properties
                dbInstanceClass: databaseInstanceType(props.instanceProps.instanceType),
            });
            instance.applyRemovalPolicy(props.removalPolicy, {
                applyToUpdateReplacePolicy: true,
            });
            // We must have a dependency on the NAT gateway provider here to create
            // things in the right order.
            instance.node.addDependency(internetConnectivityEstablished);
            this.instanceIdentifiers.push(instance.ref);
            this.instanceEndpoints.push(new endpoint_1.Endpoint(instance.attrEndpoint, port));
        }
        this.connections = new ec2.Connections({
            defaultPort: ec2.Port.tcp(port),
            securityGroups: [securityGroup],
        });
    }
    /**
     * Import an existing DatabaseCluster from properties
     */
    static fromDatabaseClusterAttributes(scope, id, attrs) {
        class Import extends DatabaseClusterBase {
            constructor() {
                super(...arguments);
                this.defaultPort = ec2.Port.tcp(attrs.port);
                this.connections = new ec2.Connections({
                    securityGroups: [attrs.securityGroup],
                    defaultPort: this.defaultPort,
                });
                this.clusterIdentifier = attrs.clusterIdentifier;
                this.instanceIdentifiers = attrs.instanceIdentifiers;
                this.clusterEndpoint = new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.port);
                this.clusterReadEndpoint = new endpoint_1.Endpoint(attrs.readerEndpointAddress, attrs.port);
                this.instanceEndpoints = attrs.instanceEndpointAddresses.map(a => new endpoint_1.Endpoint(a, attrs.port));
                this.securityGroupId = attrs.securityGroup.securityGroupId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds the single user rotation of the master password to this cluster.
     *
     * @param [automaticallyAfter=Duration.days(30)] Specifies the number of days after the previous rotation
     * before Secrets Manager triggers the next automatic rotation.
     */
    addRotationSingleUser(automaticallyAfter) {
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for a cluster without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this cluster.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            automaticallyAfter,
            application: DatabaseCluster.SINGLE_USER_ROTATION_APPLICATION,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    /**
     * Adds the multi user rotation to this cluster.
     */
    addRotationMultiUser(id, options) {
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for a cluster without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: options.secret,
            masterSecret: this.secret,
            automaticallyAfter: options.automaticallyAfter,
            application: DatabaseCluster.MULTI_USER_ROTATION_APPLICATION,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
}
exports.DatabaseCluster = DatabaseCluster;
/**
 * The default number of instances in the DocDB cluster if none are
 * specified
 */
DatabaseCluster.DEFAULT_NUM_INSTANCES = 1;
/**
 * The default port Document DB listens on
 */
DatabaseCluster.DEFAULT_PORT = 27017;
/**
 * The single user secret rotation application.
 */
DatabaseCluster.SINGLE_USER_ROTATION_APPLICATION = secretsmanager.SecretRotationApplication.MONGODB_ROTATION_SINGLE_USER;
/**
 * The multi user secret rotation application.
 */
DatabaseCluster.MULTI_USER_ROTATION_APPLICATION = secretsmanager.SecretRotationApplication.MONGODB_ROTATION_MULTI_USER;
/**
 * Turn a regular instance type into a database instance type
 */
function databaseInstanceType(instanceType) {
    return 'db.' + instanceType.toString();
}
//# sourceMappingURL=data:application/json;base64,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