"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Endpoint = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Connection endpoint of a database cluster or instance
 *
 * Consists of a combination of hostname and port.
 */
class Endpoint {
    /**
     * Constructs an Endpoint instance.
     *
     * @param address - The hostname or address of the endpoint
     * @param port - The port number of the endpoint
     */
    constructor(address, port) {
        if (!core_1.Token.isUnresolved(port) && !Endpoint.isValidPort(port)) {
            throw new Error(`Port must be an integer between [${Endpoint.MIN_PORT}, ${Endpoint.MAX_PORT}] but got: ${port}`);
        }
        this.hostname = address;
        this.port = port;
        const portDesc = core_1.Token.isUnresolved(port) ? core_1.Token.asString(port) : port;
        this.socketAddress = `${address}:${portDesc}`;
    }
    /**
     * Determines if a port is valid
     *
     * @param port: The port number
     * @returns boolean whether the port is valid
     */
    static isValidPort(port) {
        return Number.isInteger(port) && port >= Endpoint.MIN_PORT && port <= Endpoint.MAX_PORT;
    }
    /**
     * Returns the port number as a string representation that can be used for embedding within other strings.
     *
     * This is intended to deal with CDK's token system. Numeric CDK tokens are not expanded when their string
     * representation is embedded in a string. This function returns the port either as an unresolved string token or
     * as a resolved string representation of the port value.
     *
     * @returns {string} An (un)resolved string representation of the endpoint's port number
     */
    portAsString() {
        if (core_1.Token.isUnresolved(this.port)) {
            return core_1.Token.asString(this.port);
        }
        else {
            return this.port.toString();
        }
    }
}
exports.Endpoint = Endpoint;
/**
 * The minimum port value
 */
Endpoint.MIN_PORT = 1;
/**
 * The maximum port value
 */
Endpoint.MAX_PORT = 65535;
//# sourceMappingURL=data:application/json;base64,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