"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CidrBlock = exports.NetworkBuilder = exports.NetworkUtils = exports.InvalidCidrRangeError = void 0;
/**
 * InvalidCidrRangeError is thrown when attempting to perform operations on a CIDR
 * range that is either not valid, or outside of the VPC size limits.
 */
class InvalidCidrRangeError extends Error {
    constructor(cidr) {
        super(cidr + ' is not a valid VPC CIDR range (must be between /16 and /28)');
        // The following line is required for type checking of custom errors, and must be called right after super()
        // https://stackoverflow.com/questions/31626231/custom-error-class-in-typescript
        Object.setPrototypeOf(this, InvalidCidrRangeError.prototype);
    }
}
exports.InvalidCidrRangeError = InvalidCidrRangeError;
/**
 * NetworkUtils contains helpers to work with network constructs (subnets/ranges)
 */
class NetworkUtils {
    /**
     * Validates an IPv4 string
     *
     * returns true of the string contains 4 numbers between 0-255 delimited by
     * a `.` character
     */
    static validIp(ipAddress) {
        const octets = ipAddress.split('.');
        if (octets.length !== 4) {
            return false;
        }
        return octets.map((octet) => parseInt(octet, 10)).
            filter((octet) => octet >= 0 && octet <= 255).length === 4;
    }
    /**
     * Converts a string IPv4 to a number
     *
     * takes an IP Address (e.g. 174.66.173.168) and converts to a number
     * (e.g 2923605416); currently only supports IPv4
     *
     * Uses the formula:
     * (first octet * 256³) + (second octet * 256²) + (third octet * 256) +
     * (fourth octet)
     *
     * @param  {string} the IP address (e.g. 174.66.173.168)
     * @returns {number} the integer value of the IP address (e.g 2923605416)
     */
    static ipToNum(ipAddress) {
        if (!this.validIp(ipAddress)) {
            throw new Error(`${ipAddress} is not valid`);
        }
        return ipAddress
            .split('.')
            .reduce((p, c, i) => p + parseInt(c, 10) * 256 ** (3 - i), 0);
    }
    /**
     * Takes number and converts it to IPv4 address string
     *
     * Takes a number (e.g 2923605416) and converts it to an IPv4 address string
     * currently only supports IPv4
     *
     * @param  {number} the integer value of the IP address (e.g 2923605416)
     * @returns {string} the IPv4 address (e.g. 174.66.173.168)
     */
    static numToIp(ipNum) {
        // this all because bitwise math is signed
        let remaining = ipNum;
        const address = new Array();
        for (let i = 0; i < 4; i++) {
            if (remaining !== 0) {
                address.push(Math.floor(remaining / 256 ** (3 - i)));
                remaining = remaining % 256 ** (3 - i);
            }
            else {
                address.push(0);
            }
        }
        const ipAddress = address.join('.');
        if (!this.validIp(ipAddress)) {
            throw new Error(`${ipAddress} is not a valid IP Address`);
        }
        return ipAddress;
    }
}
exports.NetworkUtils = NetworkUtils;
/**
 * Creates a network based on a CIDR Block to build contained subnets
 */
class NetworkBuilder {
    /**
     * Create a network using the provided CIDR block
     *
     * No subnets are allocated in the constructor, the maxIpConsumed is set one
     * less than the first IP in the network
     *
     */
    constructor(cidr) {
        /**
         * The list of CIDR blocks for subnets within this network
         */
        this.subnetCidrs = [];
        this.networkCidr = new CidrBlock(cidr);
        this.subnetCidrs = [];
        this.nextAvailableIp = this.networkCidr.minAddress();
    }
    /**
     * Add a subnet to the network and update the maxIpConsumed
     */
    addSubnet(mask) {
        return this.addSubnets(mask, 1)[0];
    }
    /**
     * Add {count} number of subnets to the network and update the maxIpConsumed
     */
    addSubnets(mask, count = 1) {
        if (mask < 16 || mask > 28) {
            throw new InvalidCidrRangeError(`x.x.x.x/${mask}`);
        }
        const maxIp = this.nextAvailableIp + (CidrBlock.calculateNetsize(mask) * count);
        if (this.networkCidr.maxAddress() < maxIp - 1) {
            throw new Error(`${count} of /${mask} exceeds remaining space of ${this.networkCidr.cidr}`);
        }
        const subnets = [];
        for (let i = 0; i < count; i++) {
            const subnet = new CidrBlock(this.nextAvailableIp, mask);
            this.nextAvailableIp = subnet.nextBlock().minAddress();
            this.subnetCidrs.push(subnet);
            subnets.push(subnet);
        }
        return subnets.map((subnet) => (subnet.cidr));
    }
    /**
     * return the CIDR notation strings for all subnets in the network
     */
    get cidrStrings() {
        return this.subnetCidrs.map((subnet) => (subnet.cidr));
    }
    /**
     * Calculates the largest subnet to create of the given count from the
     * remaining IP space
     */
    maskForRemainingSubnets(subnetCount) {
        const remaining = this.networkCidr.maxAddress() - this.nextAvailableIp + 1;
        const ipsPerSubnet = Math.floor(remaining / subnetCount);
        return 32 - Math.floor(Math.log2(ipsPerSubnet));
    }
}
exports.NetworkBuilder = NetworkBuilder;
/**
 * A block of IP address space with a given bit prefix
 */
class CidrBlock {
    constructor(ipAddressOrCidr, mask) {
        if (typeof ipAddressOrCidr === 'string') {
            this.mask = parseInt(ipAddressOrCidr.split('/')[1], 10);
            this.networkAddress = NetworkUtils.ipToNum(ipAddressOrCidr.split('/')[0]) +
                CidrBlock.calculateNetsize(this.mask) - 1;
        }
        else {
            if (typeof mask === 'number') {
                this.mask = mask;
            }
            else {
                // this should be impossible
                this.mask = 16;
            }
            this.networkAddress = ipAddressOrCidr + CidrBlock.calculateNetsize(this.mask) - 1;
            this.networkSize = 2 ** (32 - this.mask);
        }
        this.networkSize = 2 ** (32 - this.mask);
        this.cidr = `${this.minIp()}/${this.mask}`;
    }
    /**
     * Calculates the netmask for a given CIDR mask
     *
     * For example:
     * CidrBlock.calculateNetmask(24) returns '255.255.255.0'
     */
    static calculateNetmask(mask) {
        return NetworkUtils.numToIp(2 ** 32 - 2 ** (32 - mask));
    }
    /**
     * Calculates the number IP addresses in a CIDR Mask
     *
     * For example:
     * CidrBlock.calculateNetsize(24) returns 256
     */
    static calculateNetsize(mask) {
        return 2 ** (32 - mask);
    }
    /*
     * The maximum IP in the CIDR Blcok e.g. '10.0.8.255'
     */
    maxIp() {
        // min + (2^(32-mask)) - 1 [zero needs to count]
        return NetworkUtils.numToIp(this.maxAddress());
    }
    /*
     * The minimum IP in the CIDR Blcok e.g. '10.0.0.0'
     */
    minIp() {
        return NetworkUtils.numToIp(this.minAddress());
    }
    /*
     * Returns the number representation for the minimum IPv4 address
     */
    minAddress() {
        const div = this.networkAddress % this.networkSize;
        return this.networkAddress - div;
    }
    /*
     * Returns the number representation for the maximum IPv4 address
     */
    maxAddress() {
        // min + (2^(32-mask)) - 1 [zero needs to count]
        return this.minAddress() + this.networkSize - 1;
    }
    /*
     * Returns the next CIDR Block of the same mask size
     */
    nextBlock() {
        return new CidrBlock(this.maxAddress() + 1, this.mask);
    }
    /*
     * Returns true if this CidrBlock fully contains the provided CidrBlock
     */
    containsCidr(other) {
        return (this.maxAddress() >= other.maxAddress()) &&
            (this.minAddress() <= other.minAddress());
    }
}
exports.CidrBlock = CidrBlock;
//# sourceMappingURL=data:application/json;base64,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