"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Peer = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const connections_1 = require("./connections");
/**
 * Peer object factories (to be used in Security Group management)
 *
 * The static methods on this object can be used to create peer objects
 * which represent a connection partner in Security Group rules.
 *
 * Use this object if you need to represent connection partners using plain IP
 * addresses, or a prefix list ID.
 *
 * If you want to address a connection partner by Security Group, you can just
 * use the Security Group (or the construct that contains a Security Group)
 * directly, as it already implements `IPeer`.
 */
class Peer {
    /**
     * Create an IPv4 peer from a CIDR
     */
    static ipv4(cidrIp) {
        return new CidrIPv4(cidrIp);
    }
    /**
     * Any IPv4 address
     */
    static anyIpv4() {
        return new AnyIPv4();
    }
    /**
     * Create an IPv6 peer from a CIDR
     */
    static ipv6(cidrIp) {
        return new CidrIPv6(cidrIp);
    }
    /**
     * Any IPv6 address
     */
    static anyIpv6() {
        return new AnyIPv6();
    }
    /**
     * A prefix list
     */
    static prefixList(prefixListId) {
        return new PrefixList(prefixListId);
    }
    constructor() {
    }
}
exports.Peer = Peer;
/**
 * A connection to and from a given IP range
 */
class CidrIPv4 {
    constructor(cidrIp) {
        this.cidrIp = cidrIp;
        this.canInlineRule = true;
        this.connections = new connections_1.Connections({ peer: this });
        if (!core_1.Token.isUnresolved(cidrIp)) {
            const cidrMatch = cidrIp.match(/^(\d{1,3}\.){3}\d{1,3}(\/\d+)?$/);
            if (!cidrMatch) {
                throw new Error(`Invalid IPv4 CIDR: "${cidrIp}"`);
            }
            if (!cidrMatch[2]) {
                throw new Error(`CIDR mask is missing in IPv4: "${cidrIp}". Did you mean "${cidrIp}/32"?`);
            }
        }
        this.uniqueId = cidrIp;
    }
    /**
     * Produce the ingress rule JSON for the given connection
     */
    toIngressRuleConfig() {
        return { cidrIp: this.cidrIp };
    }
    /**
     * Produce the egress rule JSON for the given connection
     */
    toEgressRuleConfig() {
        return { cidrIp: this.cidrIp };
    }
}
/**
 * Any IPv4 address
 */
class AnyIPv4 extends CidrIPv4 {
    constructor() {
        super('0.0.0.0/0');
    }
}
/**
 * A connection to a from a given IPv6 range
 */
class CidrIPv6 {
    constructor(cidrIpv6) {
        this.cidrIpv6 = cidrIpv6;
        this.canInlineRule = true;
        this.connections = new connections_1.Connections({ peer: this });
        if (!core_1.Token.isUnresolved(cidrIpv6)) {
            const cidrMatch = cidrIpv6.match(/^([\da-f]{0,4}:){2,7}([\da-f]{0,4})?(\/\d+)?$/);
            if (!cidrMatch) {
                throw new Error(`Invalid IPv6 CIDR: "${cidrIpv6}"`);
            }
            if (!cidrMatch[3]) {
                throw new Error(`CIDR mask is missing in IPv6: "${cidrIpv6}". Did you mean "${cidrIpv6}/128"?`);
            }
        }
        this.uniqueId = cidrIpv6;
    }
    /**
     * Produce the ingress rule JSON for the given connection
     */
    toIngressRuleConfig() {
        return { cidrIpv6: this.cidrIpv6 };
    }
    /**
     * Produce the egress rule JSON for the given connection
     */
    toEgressRuleConfig() {
        return { cidrIpv6: this.cidrIpv6 };
    }
}
/**
 * Any IPv6 address
 */
class AnyIPv6 extends CidrIPv6 {
    constructor() {
        super('::/0');
    }
}
/**
 * A prefix list
 *
 * Prefix lists are used to allow traffic to VPC-local service endpoints.
 *
 * For more information, see this page:
 *
 * https://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/vpc-endpoints.html
 */
class PrefixList {
    constructor(prefixListId) {
        this.prefixListId = prefixListId;
        this.canInlineRule = false;
        this.connections = new connections_1.Connections({ peer: this });
        this.uniqueId = prefixListId;
    }
    toIngressRuleConfig() {
        return { sourcePrefixListId: this.prefixListId };
    }
    toEgressRuleConfig() {
        return { destinationPrefixListId: this.prefixListId };
    }
}
//# sourceMappingURL=data:application/json;base64,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