"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Port = exports.Protocol = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Protocol for use in Connection Rules
 */
var Protocol;
(function (Protocol) {
    Protocol["ALL"] = "-1";
    Protocol["TCP"] = "tcp";
    Protocol["UDP"] = "udp";
    Protocol["ICMP"] = "icmp";
    Protocol["ICMPV6"] = "58";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
/**
 * Interface for classes that provide the connection-specification parts of a security group rule
 */
class Port {
    constructor(props) {
        this.props = props;
        this.canInlineRule = !core_1.Token.isUnresolved(props.fromPort) && !core_1.Token.isUnresolved(props.toPort);
    }
    /**
     * A single TCP port
     */
    static tcp(port) {
        return new Port({
            protocol: Protocol.TCP,
            fromPort: port,
            toPort: port,
            stringRepresentation: renderPort(port),
        });
    }
    /**
     * A TCP port range
     */
    static tcpRange(startPort, endPort) {
        return new Port({
            protocol: Protocol.TCP,
            fromPort: startPort,
            toPort: endPort,
            stringRepresentation: `${renderPort(startPort)}-${renderPort(endPort)}`,
        });
    }
    /**
     * Any TCP traffic
     */
    static allTcp() {
        return new Port({
            protocol: Protocol.TCP,
            fromPort: 0,
            toPort: 65535,
            stringRepresentation: 'ALL PORTS',
        });
    }
    /**
     * A single UDP port
     */
    static udp(port) {
        return new Port({
            protocol: Protocol.UDP,
            fromPort: port,
            toPort: port,
            stringRepresentation: `UDP ${renderPort(port)}`,
        });
    }
    /**
     * A UDP port range
     */
    static udpRange(startPort, endPort) {
        return new Port({
            protocol: Protocol.UDP,
            fromPort: startPort,
            toPort: endPort,
            stringRepresentation: `UDP ${renderPort(startPort)}-${renderPort(endPort)}`,
        });
    }
    /**
     * Any UDP traffic
     */
    static allUdp() {
        return new Port({
            protocol: Protocol.UDP,
            fromPort: 0,
            toPort: 65535,
            stringRepresentation: 'UDP ALL PORTS',
        });
    }
    /**
     * A specific combination of ICMP type and code
     *
     * @see https://www.iana.org/assignments/icmp-parameters/icmp-parameters.xhtml
     */
    static icmpTypeAndCode(type, code) {
        return new Port({
            protocol: Protocol.ICMP,
            fromPort: type,
            toPort: code,
            stringRepresentation: `ICMP Type ${type} Code ${code}`,
        });
    }
    /**
     * All codes for a single ICMP type
     */
    static icmpType(type) {
        return new Port({
            protocol: Protocol.ICMP,
            fromPort: type,
            toPort: -1,
            stringRepresentation: `ICMP Type ${type}`,
        });
    }
    /**
     * ICMP ping (echo) traffic
     */
    static icmpPing() {
        return Port.icmpType(8);
    }
    /**
     * All ICMP traffic
     */
    static allIcmp() {
        return new Port({
            protocol: Protocol.ICMP,
            fromPort: -1,
            toPort: -1,
            stringRepresentation: 'ALL ICMP',
        });
    }
    /**
     * All traffic
     */
    static allTraffic() {
        return new Port({
            protocol: Protocol.ALL,
            stringRepresentation: 'ALL TRAFFIC',
        });
    }
    /**
     * Produce the ingress/egress rule JSON for the given connection
     */
    toRuleJson() {
        return {
            ipProtocol: this.props.protocol,
            fromPort: this.props.fromPort,
            toPort: this.props.toPort,
        };
    }
    toString() {
        return this.props.stringRepresentation;
    }
}
exports.Port = Port;
function renderPort(port) {
    return core_1.Token.isUnresolved(port) ? '{IndirectPort}' : port.toString();
}
//# sourceMappingURL=data:application/json;base64,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