"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecurityGroup = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const SECURITY_GROUP_SYMBOL = Symbol.for('@aws-cdk/iam.SecurityGroup');
/**
 * A SecurityGroup that is not created in this template
 */
class SecurityGroupBase extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.canInlineRule = false;
        this.connections = new connections_1.Connections({ securityGroups: [this] });
        Object.defineProperty(this, SECURITY_GROUP_SYMBOL, { value: true });
    }
    /**
     * Return whether the indicated object is a security group
     */
    static isSecurityGroup(x) {
        return SECURITY_GROUP_SYMBOL in x;
    }
    get uniqueId() {
        return this.node.uniqueId;
    }
    addIngressRule(peer, connection, description, remoteRule) {
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        const [scope, id] = determineRuleScope(this, peer, connection, 'from', remoteRule);
        // Skip duplicates
        if (scope.node.tryFindChild(id) === undefined) {
            new ec2_generated_1.CfnSecurityGroupIngress(scope, id, {
                groupId: this.securityGroupId,
                ...peer.toIngressRuleConfig(),
                ...connection.toRuleJson(),
                description,
            });
        }
    }
    addEgressRule(peer, connection, description, remoteRule) {
        if (description === undefined) {
            description = `to ${peer.uniqueId}:${connection}`;
        }
        const [scope, id] = determineRuleScope(this, peer, connection, 'to', remoteRule);
        // Skip duplicates
        if (scope.node.tryFindChild(id) === undefined) {
            new ec2_generated_1.CfnSecurityGroupEgress(scope, id, {
                groupId: this.securityGroupId,
                ...peer.toEgressRuleConfig(),
                ...connection.toRuleJson(),
                description,
            });
        }
    }
    toIngressRuleConfig() {
        return { sourceSecurityGroupId: this.securityGroupId };
    }
    toEgressRuleConfig() {
        return { destinationSecurityGroupId: this.securityGroupId };
    }
}
/**
 * Determine where to parent a new ingress/egress rule
 *
 * A SecurityGroup rule is parented under the group it's related to, UNLESS
 * we're in a cross-stack scenario with another Security Group. In that case,
 * we respect the 'remoteRule' flag and will parent under the other security
 * group.
 *
 * This is necessary to avoid cyclic dependencies between stacks, since both
 * ingress and egress rules will reference both security groups, and a naive
 * parenting will lead to the following situation:
 *
 *   ╔════════════════════╗         ╔════════════════════╗
 *   ║  ┌───────────┐     ║         ║    ┌───────────┐   ║
 *   ║  │  GroupA   │◀────╬─┐   ┌───╬───▶│  GroupB   │   ║
 *   ║  └───────────┘     ║ │   │   ║    └───────────┘   ║
 *   ║        ▲           ║ │   │   ║          ▲         ║
 *   ║        │           ║ │   │   ║          │         ║
 *   ║        │           ║ │   │   ║          │         ║
 *   ║  ┌───────────┐     ║ └───┼───╬────┌───────────┐   ║
 *   ║  │  EgressA  │─────╬─────┘   ║    │ IngressB  │   ║
 *   ║  └───────────┘     ║         ║    └───────────┘   ║
 *   ║                    ║         ║                    ║
 *   ╚════════════════════╝         ╚════════════════════╝
 *
 * By having the ability to switch the parent, we avoid the cyclic reference by
 * keeping all rules in a single stack.
 *
 * If this happens, we also have to change the construct ID, because
 * otherwise we might have two objects with the same ID if we have
 * multiple reversed security group relationships.
 *
 *   ╔═══════════════════════════════════╗
 *   ║┌───────────┐                      ║
 *   ║│  GroupB   │                      ║
 *   ║└───────────┘                      ║
 *   ║      ▲                            ║
 *   ║      │              ┌───────────┐ ║
 *   ║      ├────"from A"──│ IngressB  │ ║
 *   ║      │              └───────────┘ ║
 *   ║      │              ┌───────────┐ ║
 *   ║      ├─────"to B"───│  EgressA  │ ║
 *   ║      │              └───────────┘ ║
 *   ║      │              ┌───────────┐ ║
 *   ║      └─────"to B"───│  EgressC  │ ║  <-- oops
 *   ║                     └───────────┘ ║
 *   ╚═══════════════════════════════════╝
 */
function determineRuleScope(group, peer, connection, fromTo, remoteRule) {
    if (remoteRule && SecurityGroupBase.isSecurityGroup(peer) && differentStacks(group, peer)) {
        // Reversed
        const reversedFromTo = fromTo === 'from' ? 'to' : 'from';
        return [peer, `${group.uniqueId}:${connection} ${reversedFromTo}`];
    }
    else {
        // Regular (do old ID escaping to in order to not disturb existing deployments)
        return [group, `${fromTo} ${renderPeer(peer)}:${connection}`.replace('/', '_')];
    }
}
function renderPeer(peer) {
    return core_1.Token.isUnresolved(peer.uniqueId) ? '{IndirectPeer}' : peer.uniqueId;
}
function differentStacks(group1, group2) {
    return core_1.Stack.of(group1) !== core_1.Stack.of(group2);
}
/**
 * Creates an Amazon EC2 security group within a VPC.
 *
 * Security Groups act like a firewall with a set of rules, and are associated
 * with any AWS resource that has or creates Elastic Network Interfaces (ENIs).
 * A typical example of a resource that has a security group is an Instance (or
 * Auto Scaling Group of instances)
 *
 * If you are defining new infrastructure in CDK, there is a good chance you
 * won't have to interact with this class at all. Like IAM Roles, Security
 * Groups need to exist to control access between AWS resources, but CDK will
 * automatically generate and populate them with least-privilege permissions
 * for you so you can concentrate on your business logic.
 *
 * All Constructs that require Security Groups will create one for you if you
 * don't specify one at construction. After construction, you can selectively
 * allow connections to and between constructs via--for example-- the `instance.connections`
 * object. Think of it as "allowing connections to your instance", rather than
 * "adding ingress rules a security group". See the [Allowing
 * Connections](https://docs.aws.amazon.com/cdk/api/latest/docs/aws-ec2-readme.html#allowing-connections)
 * section in the library documentation for examples.
 *
 * Direct manipulation of the Security Group through `addIngressRule` and
 * `addEgressRule` is possible, but mutation through the `.connections` object
 * is recommended. If you peer two constructs with security groups this way,
 * appropriate rules will be created in both.
 *
 * If you have an existing security group you want to use in your CDK application,
 * you would import it like this:
 *
 * ```ts
 * const securityGroup = SecurityGroup.fromSecurityGroupId(this, 'SG', 'sg-12345', {
 *   mutable: false
 * });
 * ```
 */
class SecurityGroup extends SecurityGroupBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.securityGroupName,
        });
        this.directIngressRules = [];
        this.directEgressRules = [];
        const groupDescription = props.description || this.node.path;
        this.allowAllOutbound = props.allowAllOutbound !== false;
        this.securityGroup = new ec2_generated_1.CfnSecurityGroup(this, 'Resource', {
            groupName: this.physicalName,
            groupDescription,
            securityGroupIngress: core_1.Lazy.anyValue({ produce: () => this.directIngressRules }, { omitEmptyArray: true }),
            securityGroupEgress: core_1.Lazy.anyValue({ produce: () => this.directEgressRules }, { omitEmptyArray: true }),
            vpcId: props.vpc.vpcId,
        });
        this.securityGroupId = this.securityGroup.attrGroupId;
        this.securityGroupVpcId = this.securityGroup.attrVpcId;
        this.securityGroupName = this.securityGroup.ref;
        this.addDefaultEgressRule();
    }
    /**
     * Import an existing security group into this app.
     */
    static fromSecurityGroupId(scope, id, securityGroupId, options = {}) {
        class MutableImport extends SecurityGroupBase {
            constructor() {
                var _a;
                super(...arguments);
                this.securityGroupId = securityGroupId;
                this.allowAllOutbound = (_a = options.allowAllOutbound) !== null && _a !== void 0 ? _a : true;
            }
            addEgressRule(peer, connection, description, remoteRule) {
                // Only if allowAllOutbound has been disabled
                if (options.allowAllOutbound === false) {
                    super.addEgressRule(peer, connection, description, remoteRule);
                }
            }
        }
        class ImmutableImport extends SecurityGroupBase {
            constructor() {
                var _a;
                super(...arguments);
                this.securityGroupId = securityGroupId;
                this.allowAllOutbound = (_a = options.allowAllOutbound) !== null && _a !== void 0 ? _a : true;
            }
            addEgressRule(_peer, _connection, _description, _remoteRule) {
                // do nothing
            }
            addIngressRule(_peer, _connection, _description, _remoteRule) {
                // do nothing
            }
        }
        return options.mutable !== false
            ? new MutableImport(scope, id)
            : new ImmutableImport(scope, id);
    }
    addIngressRule(peer, connection, description, remoteRule) {
        if (!peer.canInlineRule || !connection.canInlineRule) {
            super.addIngressRule(peer, connection, description, remoteRule);
            return;
        }
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        this.addDirectIngressRule({
            ...peer.toIngressRuleConfig(),
            ...connection.toRuleJson(),
            description,
        });
    }
    addEgressRule(peer, connection, description, remoteRule) {
        if (this.allowAllOutbound) {
            // In the case of "allowAllOutbound", we don't add any more rules. There
            // is only one rule which allows all traffic and that subsumes any other
            // rule.
            core_1.Annotations.of(this).addWarning('Ignoring Egress rule since \'allowAllOutbound\' is set to true; To add customize rules, set allowAllOutbound=false on the SecurityGroup');
            return;
        }
        else {
            // Otherwise, if the bogus rule exists we can now remove it because the
            // presence of any other rule will get rid of EC2's implicit "all
            // outbound" rule anyway.
            this.removeNoTrafficRule();
        }
        if (!peer.canInlineRule || !connection.canInlineRule) {
            super.addEgressRule(peer, connection, description, remoteRule);
            return;
        }
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        const rule = {
            ...peer.toEgressRuleConfig(),
            ...connection.toRuleJson(),
            description,
        };
        if (isAllTrafficRule(rule)) {
            // We cannot allow this; if someone adds the rule in this way, it will be
            // removed again if they add other rules. We also can't automatically switch
            // to "allOutbound=true" mode, because we might have already emitted
            // EgressRule objects (which count as rules added later) and there's no way
            // to recall those. Better to prevent this for now.
            throw new Error('Cannot add an "all traffic" egress rule in this way; set allowAllOutbound=true on the SecurityGroup instead.');
        }
        this.addDirectEgressRule(rule);
    }
    /**
     * Add a direct ingress rule
     */
    addDirectIngressRule(rule) {
        if (!this.hasIngressRule(rule)) {
            this.directIngressRules.push(rule);
        }
    }
    /**
     * Return whether the given ingress rule exists on the group
     */
    hasIngressRule(rule) {
        return this.directIngressRules.findIndex(r => ingressRulesEqual(r, rule)) > -1;
    }
    /**
     * Add a direct egress rule
     */
    addDirectEgressRule(rule) {
        if (!this.hasEgressRule(rule)) {
            this.directEgressRules.push(rule);
        }
    }
    /**
     * Return whether the given egress rule exists on the group
     */
    hasEgressRule(rule) {
        return this.directEgressRules.findIndex(r => egressRulesEqual(r, rule)) > -1;
    }
    /**
     * Add the default egress rule to the securityGroup
     *
     * This depends on allowAllOutbound:
     *
     * - If allowAllOutbound is true, we *TECHNICALLY* don't need to do anything, because
     *   EC2 is going to create this default rule anyway. But, for maximum readability
     *   of the template, we will add one anyway.
     * - If allowAllOutbound is false, we add a bogus rule that matches no traffic in
     *   order to get rid of the default "all outbound" rule that EC2 creates by default.
     *   If other rules happen to get added later, we remove the bogus rule again so
     *   that it doesn't clutter up the template too much (even though that's not
     *   strictly necessary).
     */
    addDefaultEgressRule() {
        if (this.allowAllOutbound) {
            this.directEgressRules.push(ALLOW_ALL_RULE);
        }
        else {
            this.directEgressRules.push(MATCH_NO_TRAFFIC);
        }
    }
    /**
     * Remove the bogus rule if it exists
     */
    removeNoTrafficRule() {
        const i = this.directEgressRules.findIndex(r => egressRulesEqual(r, MATCH_NO_TRAFFIC));
        if (i > -1) {
            this.directEgressRules.splice(i, 1);
        }
    }
}
exports.SecurityGroup = SecurityGroup;
/**
 * Egress rule that purposely matches no traffic
 *
 * This is used in order to disable the "all traffic" default of Security Groups.
 *
 * No machine can ever actually have the 255.255.255.255 IP address, but
 * in order to lock it down even more we'll restrict to a nonexistent
 * ICMP traffic type.
 */
const MATCH_NO_TRAFFIC = {
    cidrIp: '255.255.255.255/32',
    description: 'Disallow all traffic',
    ipProtocol: 'icmp',
    fromPort: 252,
    toPort: 86,
};
/**
 * Egress rule that matches all traffic
 */
const ALLOW_ALL_RULE = {
    cidrIp: '0.0.0.0/0',
    description: 'Allow all outbound traffic by default',
    ipProtocol: '-1',
};
/**
 * Compare two ingress rules for equality the same way CloudFormation would (discarding description)
 */
function ingressRulesEqual(a, b) {
    return a.cidrIp === b.cidrIp
        && a.cidrIpv6 === b.cidrIpv6
        && a.fromPort === b.fromPort
        && a.toPort === b.toPort
        && a.ipProtocol === b.ipProtocol
        && a.sourceSecurityGroupId === b.sourceSecurityGroupId
        && a.sourceSecurityGroupName === b.sourceSecurityGroupName
        && a.sourceSecurityGroupOwnerId === b.sourceSecurityGroupOwnerId;
}
/**
 * Compare two egress rules for equality the same way CloudFormation would (discarding description)
 */
function egressRulesEqual(a, b) {
    return a.cidrIp === b.cidrIp
        && a.cidrIpv6 === b.cidrIpv6
        && a.fromPort === b.fromPort
        && a.toPort === b.toPort
        && a.ipProtocol === b.ipProtocol
        && a.destinationPrefixListId === b.destinationPrefixListId
        && a.destinationSecurityGroupId === b.destinationSecurityGroupId;
}
/**
 * Whether this rule refers to all traffic
 */
function isAllTrafficRule(rule) {
    return rule.cidrIp === '0.0.0.0/0' && rule.ipProtocol === '-1';
}
//# sourceMappingURL=data:application/json;base64,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