"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationLoadBalancedEc2Service = void 0;
const aws_ecs_1 = require("../../../aws-ecs"); // Automatically re-written from '@aws-cdk/aws-ecs'
const application_load_balanced_service_base_1 = require("../base/application-load-balanced-service-base");
/**
 * An EC2 service running on an ECS cluster fronted by an application load balancer.
 */
class ApplicationLoadBalancedEc2Service extends application_load_balanced_service_base_1.ApplicationLoadBalancedServiceBase {
    /**
     * Constructs a new instance of the ApplicationLoadBalancedEc2Service class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        if (props.taskDefinition && props.taskImageOptions) {
            throw new Error('You must specify either a taskDefinition or taskImageOptions, not both.');
        }
        else if (props.taskDefinition) {
            this.taskDefinition = props.taskDefinition;
        }
        else if (props.taskImageOptions) {
            const taskImageOptions = props.taskImageOptions;
            this.taskDefinition = new aws_ecs_1.Ec2TaskDefinition(this, 'TaskDef', {
                executionRole: taskImageOptions.executionRole,
                taskRole: taskImageOptions.taskRole,
                family: taskImageOptions.family,
            });
            // Create log driver if logging is enabled
            const enableLogging = taskImageOptions.enableLogging !== undefined ? taskImageOptions.enableLogging : true;
            const logDriver = taskImageOptions.logDriver !== undefined
                ? taskImageOptions.logDriver : enableLogging
                ? this.createAWSLogDriver(this.node.id) : undefined;
            const containerName = taskImageOptions.containerName !== undefined ? taskImageOptions.containerName : 'web';
            const container = this.taskDefinition.addContainer(containerName, {
                image: taskImageOptions.image,
                cpu: props.cpu,
                memoryLimitMiB: props.memoryLimitMiB,
                memoryReservationMiB: props.memoryReservationMiB,
                environment: taskImageOptions.environment,
                secrets: taskImageOptions.secrets,
                logging: logDriver,
            });
            container.addPortMappings({
                containerPort: taskImageOptions.containerPort || 80,
            });
        }
        else {
            throw new Error('You must specify one of: taskDefinition or image');
        }
        this.service = new aws_ecs_1.Ec2Service(this, 'Service', {
            cluster: this.cluster,
            desiredCount: this.desiredCount,
            taskDefinition: this.taskDefinition,
            assignPublicIp: false,
            serviceName: props.serviceName,
            healthCheckGracePeriod: props.healthCheckGracePeriod,
            minHealthyPercent: props.minHealthyPercent,
            maxHealthyPercent: props.maxHealthyPercent,
            propagateTags: props.propagateTags,
            enableECSManagedTags: props.enableECSManagedTags,
            cloudMapOptions: props.cloudMapOptions,
        });
        this.addServiceAsTarget(this.service);
    }
}
exports.ApplicationLoadBalancedEc2Service = ApplicationLoadBalancedEc2Service;
//# sourceMappingURL=data:application/json;base64,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