"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationLoadBalancedFargateService = void 0;
const aws_ecs_1 = require("../../../aws-ecs"); // Automatically re-written from '@aws-cdk/aws-ecs'
const application_load_balanced_service_base_1 = require("../base/application-load-balanced-service-base");
/**
 * A Fargate service running on an ECS cluster fronted by an application load balancer.
 */
class ApplicationLoadBalancedFargateService extends application_load_balanced_service_base_1.ApplicationLoadBalancedServiceBase {
    /**
     * Constructs a new instance of the ApplicationLoadBalancedFargateService class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.assignPublicIp = props.assignPublicIp !== undefined ? props.assignPublicIp : false;
        if (props.taskDefinition && props.taskImageOptions) {
            throw new Error('You must specify either a taskDefinition or an image, not both.');
        }
        else if (props.taskDefinition) {
            this.taskDefinition = props.taskDefinition;
        }
        else if (props.taskImageOptions) {
            const taskImageOptions = props.taskImageOptions;
            this.taskDefinition = new aws_ecs_1.FargateTaskDefinition(this, 'TaskDef', {
                memoryLimitMiB: props.memoryLimitMiB,
                cpu: props.cpu,
                executionRole: taskImageOptions.executionRole,
                taskRole: taskImageOptions.taskRole,
                family: taskImageOptions.family,
            });
            // Create log driver if logging is enabled
            const enableLogging = taskImageOptions.enableLogging !== undefined ? taskImageOptions.enableLogging : true;
            const logDriver = taskImageOptions.logDriver !== undefined
                ? taskImageOptions.logDriver : enableLogging
                ? this.createAWSLogDriver(this.node.id) : undefined;
            const containerName = taskImageOptions.containerName !== undefined ? taskImageOptions.containerName : 'web';
            const container = this.taskDefinition.addContainer(containerName, {
                image: taskImageOptions.image,
                logging: logDriver,
                environment: taskImageOptions.environment,
                secrets: taskImageOptions.secrets,
            });
            container.addPortMappings({
                containerPort: taskImageOptions.containerPort || 80,
            });
        }
        else {
            throw new Error('You must specify one of: taskDefinition or image');
        }
        this.service = new aws_ecs_1.FargateService(this, 'Service', {
            cluster: this.cluster,
            desiredCount: this.desiredCount,
            taskDefinition: this.taskDefinition,
            assignPublicIp: this.assignPublicIp,
            serviceName: props.serviceName,
            healthCheckGracePeriod: props.healthCheckGracePeriod,
            minHealthyPercent: props.minHealthyPercent,
            maxHealthyPercent: props.maxHealthyPercent,
            propagateTags: props.propagateTags,
            enableECSManagedTags: props.enableECSManagedTags,
            cloudMapOptions: props.cloudMapOptions,
            platformVersion: props.platformVersion,
            securityGroups: props.securityGroups,
        });
        this.addServiceAsTarget(this.service);
    }
}
exports.ApplicationLoadBalancedFargateService = ApplicationLoadBalancedFargateService;
//# sourceMappingURL=data:application/json;base64,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