"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InstanceDrainHook = void 0;
const fs = require("fs");
const path = require("path");
const autoscaling = require("../../../aws-autoscaling"); // Automatically re-written from '@aws-cdk/aws-autoscaling'
const hooks = require("../../../aws-autoscaling-hooktargets"); // Automatically re-written from '@aws-cdk/aws-autoscaling-hooktargets'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * A hook to drain instances from ECS traffic before they're terminated
 */
class InstanceDrainHook extends cdk.Construct {
    /**
     * Constructs a new instance of the InstanceDrainHook class.
     */
    constructor(scope, id, props) {
        super(scope, id);
        const drainTime = props.drainTime || cdk.Duration.minutes(5);
        // Invoke Lambda via SNS Topic
        const fn = new lambda.Function(this, 'Function', {
            code: lambda.Code.fromInline(fs.readFileSync(path.join(__dirname, 'lambda-source', 'index.py'), { encoding: 'utf-8' })),
            handler: 'index.lambda_handler',
            runtime: lambda.Runtime.PYTHON_3_6,
            // Timeout: some extra margin for additional API calls made by the Lambda,
            // up to a maximum of 15 minutes.
            timeout: cdk.Duration.seconds(Math.min(drainTime.toSeconds() + 10, 900)),
            environment: {
                CLUSTER: props.cluster.clusterName,
            },
        });
        // Hook everything up: ASG -> Topic, Topic -> Lambda
        props.autoScalingGroup.addLifecycleHook('DrainHook', {
            lifecycleTransition: autoscaling.LifecycleTransition.INSTANCE_TERMINATING,
            defaultResult: autoscaling.DefaultResult.CONTINUE,
            notificationTarget: new hooks.FunctionHook(fn, props.topicEncryptionKey),
            heartbeatTimeout: drainTime,
        });
        // Describe actions cannot be restricted and restrict the CompleteLifecycleAction to the ASG arn
        // https://docs.aws.amazon.com/autoscaling/ec2/userguide/control-access-using-iam.html
        fn.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ec2:DescribeInstances',
                'ec2:DescribeInstanceAttribute',
                'ec2:DescribeInstanceStatus',
                'ec2:DescribeHosts',
            ],
            resources: ['*'],
        }));
        // Restrict to the ASG
        fn.addToRolePolicy(new iam.PolicyStatement({
            actions: ['autoscaling:CompleteLifecycleAction'],
            resources: [props.autoScalingGroup.autoScalingGroupArn],
        }));
        fn.addToRolePolicy(new iam.PolicyStatement({
            actions: ['ecs:DescribeContainerInstances', 'ecs:DescribeTasks'],
            resources: ['*'],
            conditions: {
                ArnEquals: { 'ecs:cluster': props.cluster.clusterArn },
            },
        }));
        // Restrict to the ECS Cluster
        fn.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ecs:ListContainerInstances',
                'ecs:SubmitContainerStateChange',
                'ecs:SubmitTaskStateChange',
            ],
            resources: [props.cluster.clusterArn],
        }));
        // Restrict the container-instance operations to the ECS Cluster
        fn.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ecs:UpdateContainerInstancesState',
                'ecs:ListTasks',
            ],
            conditions: {
                ArnEquals: { 'ecs:cluster': props.cluster.clusterArn },
            },
            resources: ['*'],
        }));
    }
}
exports.InstanceDrainHook = InstanceDrainHook;
//# sourceMappingURL=data:application/json;base64,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