"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FargatePlatformVersion = exports.FargateService = void 0;
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const base_service_1 = require("../base/base-service");
const from_service_attributes_1 = require("../base/from-service-attributes");
/**
 * This creates a service using the Fargate launch type on an ECS cluster.
 *
 * @resource AWS::ECS::Service
 */
class FargateService extends base_service_1.BaseService {
    /**
     * Imports from the specified service ARN.
     */
    static fromFargateServiceArn(scope, id, fargateServiceArn) {
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.serviceArn = fargateServiceArn;
                this.serviceName = cdk.Stack.of(scope).parseArn(fargateServiceArn).resourceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports from the specified service attrributes.
     */
    static fromFargateServiceAttributes(scope, id, attrs) {
        return from_service_attributes_1.fromServiceAtrributes(scope, id, attrs);
    }
    /**
     * Constructs a new instance of the FargateService class.
     */
    constructor(scope, id, props) {
        var _a;
        if (!props.taskDefinition.isFargateCompatible) {
            throw new Error('Supplied TaskDefinition is not configured for compatibility with Fargate');
        }
        if (props.propagateTags && props.propagateTaskTagsFrom) {
            throw new Error('You can only specify either propagateTags or propagateTaskTagsFrom. Alternatively, you can leave both blank');
        }
        if (props.securityGroup !== undefined && props.securityGroups !== undefined) {
            throw new Error('Only one of SecurityGroup or SecurityGroups can be populated.');
        }
        const propagateTagsFromSource = props.propagateTaskTagsFrom !== undefined ? props.propagateTaskTagsFrom
            : (props.propagateTags !== undefined ? props.propagateTags : base_service_1.PropagatedTagSource.NONE);
        super(scope, id, {
            ...props,
            desiredCount: props.desiredCount !== undefined ? props.desiredCount : 1,
            launchType: base_service_1.LaunchType.FARGATE,
            propagateTags: propagateTagsFromSource,
            enableECSManagedTags: props.enableECSManagedTags,
        }, {
            cluster: props.cluster.clusterName,
            taskDefinition: ((_a = props.deploymentController) === null || _a === void 0 ? void 0 : _a.type) === base_service_1.DeploymentControllerType.EXTERNAL ? undefined : props.taskDefinition.taskDefinitionArn,
            platformVersion: props.platformVersion,
        }, props.taskDefinition);
        let securityGroups;
        if (props.securityGroup !== undefined) {
            securityGroups = [props.securityGroup];
        }
        else if (props.securityGroups !== undefined) {
            securityGroups = props.securityGroups;
        }
        this.configureAwsVpcNetworkingWithSecurityGroups(props.cluster.vpc, props.assignPublicIp, props.vpcSubnets, securityGroups);
        if (!props.taskDefinition.defaultContainer) {
            throw new Error('A TaskDefinition must have at least one essential container');
        }
    }
}
exports.FargateService = FargateService;
/**
 * The platform version on which to run your service.
 *
 * @see https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html
 */
var FargatePlatformVersion;
(function (FargatePlatformVersion) {
    /**
     * The latest, recommended platform version.
     */
    FargatePlatformVersion["LATEST"] = "LATEST";
    /**
     * Version 1.4.0
     *
     * Supports EFS endpoints, CAP_SYS_PTRACE Linux capability,
     * network performance metrics in CloudWatch Container Insights,
     * consolidated 20 GB ephemeral volume.
     */
    FargatePlatformVersion["VERSION1_4"] = "1.4.0";
    /**
     * Version 1.3.0
     *
     * Supports secrets, task recycling.
     */
    FargatePlatformVersion["VERSION1_3"] = "1.3.0";
    /**
     * Version 1.2.0
     *
     * Supports private registries.
     */
    FargatePlatformVersion["VERSION1_2"] = "1.2.0";
    /**
     * Version 1.1.0
     *
     * Supports task metadata, health checks, service discovery.
     */
    FargatePlatformVersion["VERSION1_1"] = "1.1.0";
    /**
     * Initial release
     *
     * Based on Amazon Linux 2017.09.
     */
    FargatePlatformVersion["VERSION1_0"] = "1.0.0";
})(FargatePlatformVersion = exports.FargatePlatformVersion || (exports.FargatePlatformVersion = {}));
//# sourceMappingURL=data:application/json;base64,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