"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Nodegroup = exports.NodegroupAmiType = void 0;
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cluster_1 = require("./cluster");
const eks_generated_1 = require("./eks.generated");
const instance_types_1 = require("./instance-types");
/**
 * The AMI type for your node group. GPU instance types should use the `AL2_x86_64_GPU` AMI type, which uses the
 * Amazon EKS-optimized Linux AMI with GPU support. Non-GPU instances should use the `AL2_x86_64` AMI type, which
 * uses the Amazon EKS-optimized Linux AMI.
 */
var NodegroupAmiType;
(function (NodegroupAmiType) {
    /**
     * Amazon Linux 2 (x86-64)
     */
    NodegroupAmiType["AL2_X86_64"] = "AL2_x86_64";
    /**
     * Amazon Linux 2 with GPU support
     */
    NodegroupAmiType["AL2_X86_64_GPU"] = "AL2_x86_64_GPU";
    /**
     * Amazon Linux 2 (ARM-64)
     */
    NodegroupAmiType["AL2_ARM_64"] = "AL2_ARM_64";
})(NodegroupAmiType = exports.NodegroupAmiType || (exports.NodegroupAmiType = {}));
/**
 * The Nodegroup resource class
 */
class Nodegroup extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f;
        super(scope, id, {
            physicalName: props.nodegroupName,
        });
        this.cluster = props.cluster;
        this.desiredSize = (_b = (_a = props.desiredSize) !== null && _a !== void 0 ? _a : props.minSize) !== null && _b !== void 0 ? _b : 2;
        this.maxSize = (_c = props.maxSize) !== null && _c !== void 0 ? _c : this.desiredSize;
        this.minSize = (_d = props.minSize) !== null && _d !== void 0 ? _d : 1;
        if (this.desiredSize > this.maxSize) {
            throw new Error(`Desired capacity ${this.desiredSize} can't be greater than max size ${this.maxSize}`);
        }
        if (this.desiredSize < this.minSize) {
            throw new Error(`Minimum capacity ${this.minSize} can't be greater than desired size ${this.desiredSize}`);
        }
        if (!props.nodeRole) {
            const ngRole = new aws_iam_1.Role(this, 'NodeGroupRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
            });
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
            this.role = ngRole;
        }
        else {
            this.role = props.nodeRole;
        }
        const resource = new eks_generated_1.CfnNodegroup(this, 'Resource', {
            clusterName: this.cluster.clusterName,
            nodegroupName: props.nodegroupName,
            nodeRole: this.role.roleArn,
            subnets: this.cluster.vpc.selectSubnets(props.subnets).subnetIds,
            amiType: (_e = props.amiType) !== null && _e !== void 0 ? _e : (props.instanceType ? getAmiTypeForInstanceType(props.instanceType).toString() :
                undefined),
            diskSize: props.diskSize,
            forceUpdateEnabled: (_f = props.forceUpdate) !== null && _f !== void 0 ? _f : true,
            instanceTypes: props.instanceType ? [props.instanceType.toString()] : undefined,
            labels: props.labels,
            releaseVersion: props.releaseVersion,
            remoteAccess: props.remoteAccess ? {
                ec2SshKey: props.remoteAccess.sshKeyName,
                sourceSecurityGroups: props.remoteAccess.sourceSecurityGroups ?
                    props.remoteAccess.sourceSecurityGroups.map(m => m.securityGroupId) : undefined,
            } : undefined,
            scalingConfig: {
                desiredSize: this.desiredSize,
                maxSize: this.maxSize,
                minSize: this.minSize,
            },
            tags: props.tags,
        });
        if (props.launchTemplateSpec) {
            if (props.diskSize) {
                // see - https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html
                // and https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-disksize
                throw new Error('diskSize must be specified within the launch template');
            }
            if (props.instanceType) {
                // see - https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html
                // and https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-disksize
                throw new Error('Instance types must be specified within the launch template');
            }
            // TODO: update this when the L1 resource spec is updated.
            resource.addPropertyOverride('LaunchTemplate', {
                Id: props.launchTemplateSpec.id,
                Version: props.launchTemplateSpec.version,
            });
        }
        // managed nodegroups update the `aws-auth` on creation, but we still need to track
        // its state for consistency.
        if (this.cluster instanceof cluster_1.Cluster) {
            // see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
            this.cluster.awsAuth.addRoleMapping(this.role, {
                username: 'system:node:{{EC2PrivateDNSName}}',
                groups: [
                    'system:bootstrappers',
                    'system:nodes',
                ],
            });
        }
        this.nodegroupArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'eks',
            resource: 'nodegroup',
            resourceName: this.physicalName,
        });
        this.nodegroupName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * Import the Nodegroup from attributes
     */
    static fromNodegroupName(scope, id, nodegroupName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.nodegroupName = nodegroupName;
            }
        }
        return new Import(scope, id);
    }
}
exports.Nodegroup = Nodegroup;
function getAmiTypeForInstanceType(instanceType) {
    return instance_types_1.INSTANCE_TYPES.graviton2.includes(instanceType.toString().substring(0, 3)) ? NodegroupAmiType.AL2_ARM_64 :
        instance_types_1.INSTANCE_TYPES.graviton.includes(instanceType.toString().substring(0, 2)) ? NodegroupAmiType.AL2_ARM_64 :
            instance_types_1.INSTANCE_TYPES.gpu.includes(instanceType.toString().substring(0, 2)) ? NodegroupAmiType.AL2_X86_64_GPU :
                instance_types_1.INSTANCE_TYPES.inferentia.includes(instanceType.toString().substring(0, 4)) ? NodegroupAmiType.AL2_X86_64_GPU :
                    NodegroupAmiType.AL2_X86_64;
}
//# sourceMappingURL=data:application/json;base64,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