"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateRedirectResponse = exports.validateFixedResponse = exports.ApplicationListenerRule = exports.ContentType = void 0;
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const application_listener_action_1 = require("./application-listener-action");
/**
 * The content type for a fixed response
 */
var ContentType;
(function (ContentType) {
    ContentType["TEXT_PLAIN"] = "text/plain";
    ContentType["TEXT_CSS"] = "text/css";
    ContentType["TEXT_HTML"] = "text/html";
    ContentType["APPLICATION_JAVASCRIPT"] = "application/javascript";
    ContentType["APPLICATION_JSON"] = "application/json";
})(ContentType = exports.ContentType || (exports.ContentType = {}));
/**
 * Define a new listener rule
 */
class ApplicationListenerRule extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.legacyConditions = {};
        this.conditions = props.conditions || [];
        const hasPathPatterns = props.pathPatterns || props.pathPattern;
        if (this.conditions.length === 0 && !props.hostHeader && !hasPathPatterns) {
            throw new Error('At least one of \'conditions\', \'hostHeader\', \'pathPattern\' or \'pathPatterns\' is required when defining a load balancing rule.');
        }
        const possibleActions = ['action', 'targetGroups', 'fixedResponse', 'redirectResponse'];
        const providedActions = possibleActions.filter(action => props[action] !== undefined);
        if (providedActions.length > 1) {
            throw new Error(`'${providedActions}' specified together, specify only one`);
        }
        if (props.priority <= 0) {
            throw new Error('Priority must have value greater than or equal to 1');
        }
        this.listener = props.listener;
        const resource = new elasticloadbalancingv2_generated_1.CfnListenerRule(this, 'Resource', {
            listenerArn: props.listener.listenerArn,
            priority: props.priority,
            conditions: cdk.Lazy.anyValue({ produce: () => this.renderConditions() }),
            actions: cdk.Lazy.anyValue({ produce: () => this.action ? this.action.renderActions() : [] }),
        });
        if (props.hostHeader) {
            this.setCondition('host-header', [props.hostHeader]);
        }
        if (hasPathPatterns) {
            if (props.pathPattern && props.pathPatterns) {
                throw new Error('Both `pathPatterns` and `pathPattern` are specified, specify only one');
            }
            const pathPattern = props.pathPattern ? [props.pathPattern] : props.pathPatterns;
            this.setCondition('path-pattern', pathPattern);
        }
        if (props.action) {
            this.configureAction(props.action);
        }
        (props.targetGroups || []).forEach(this.addTargetGroup.bind(this));
        if (props.fixedResponse) {
            this.addFixedResponse(props.fixedResponse);
        }
        else if (props.redirectResponse) {
            this.addRedirectResponse(props.redirectResponse);
        }
        this.listenerRuleArn = resource.ref;
    }
    /**
     * Add a non-standard condition to this rule
     *
     * If the condition conflicts with an already set condition, it will be overwritten by the one you specified.
     *
     * @deprecated use `addCondition` instead.
     */
    setCondition(field, values) {
        if (values === undefined) {
            delete this.legacyConditions[field];
            return;
        }
        this.legacyConditions[field] = values;
    }
    /**
     * Add a non-standard condition to this rule
     */
    addCondition(condition) {
        this.conditions.push(condition);
    }
    /**
     * Configure the action to perform for this rule
     */
    configureAction(action) {
        // It might make sense to 'throw' here.
        //
        // However, programs may already exist out there which configured an action twice,
        // in which case the second action accidentally overwrite the initial action, and in some
        // way ended up with a program that did what the author intended. If we were to add throw now,
        // the previously working program would be broken.
        //
        // Instead, signal this through a warning.
        // @deprecate: upon the next major version bump, replace this with a `throw`
        if (this.action) {
            cdk.Annotations.of(this).addWarning('An Action already existed on this ListenerRule and was replaced. Configure exactly one default Action.');
        }
        action.bind(this, this.listener, this);
        this.action = action;
    }
    /**
     * Add a TargetGroup to load balance to
     *
     * @deprecated Use configureAction instead
     */
    addTargetGroup(targetGroup) {
        this.configureAction(application_listener_action_1.ListenerAction.forward([targetGroup]));
    }
    /**
     * Add a fixed response
     *
     * @deprecated Use configureAction instead
     */
    addFixedResponse(fixedResponse) {
        validateFixedResponse(fixedResponse);
        this.configureAction(application_listener_action_1.ListenerAction.fixedResponse(cdk.Token.asNumber(fixedResponse.statusCode), {
            contentType: fixedResponse.contentType,
            messageBody: fixedResponse.messageBody,
        }));
    }
    /**
     * Add a redirect response
     *
     * @deprecated Use configureAction instead
     */
    addRedirectResponse(redirectResponse) {
        validateRedirectResponse(redirectResponse);
        this.configureAction(application_listener_action_1.ListenerAction.redirect({
            host: redirectResponse.host,
            path: redirectResponse.path,
            permanent: redirectResponse.statusCode === 'HTTP_301',
            port: redirectResponse.port,
            protocol: redirectResponse.protocol,
            query: redirectResponse.query,
        }));
    }
    /**
     * Validate the rule
     */
    validate() {
        if (this.action === undefined) {
            return ['Listener rule needs at least one action'];
        }
        const legacyConditionFields = Object.keys(this.legacyConditions);
        if (legacyConditionFields.length === 0 && this.conditions.length === 0) {
            return ['Listener rule needs at least one condition'];
        }
        return [];
    }
    /**
     * Render the conditions for this rule
     */
    renderConditions() {
        const legacyConditions = Object.entries(this.legacyConditions).map(([field, values]) => {
            return { field, values };
        });
        const conditions = this.conditions.map(condition => condition.renderRawCondition());
        return [
            ...legacyConditions,
            ...conditions,
        ];
    }
}
exports.ApplicationListenerRule = ApplicationListenerRule;
/**
 * Validate the status code and message body of a fixed response
 *
 * @internal
 */
function validateFixedResponse(fixedResponse) {
    if (fixedResponse.statusCode && !/^(2|4|5)\d\d$/.test(fixedResponse.statusCode)) {
        throw new Error('`statusCode` must be 2XX, 4XX or 5XX.');
    }
    if (fixedResponse.messageBody && fixedResponse.messageBody.length > 1024) {
        throw new Error('`messageBody` cannot have more than 1024 characters.');
    }
}
exports.validateFixedResponse = validateFixedResponse;
/**
 * Validate the status code and message body of a redirect response
 *
 * @internal
 */
function validateRedirectResponse(redirectResponse) {
    if (redirectResponse.protocol && !/^(HTTPS?|#\{protocol\})$/i.test(redirectResponse.protocol)) {
        throw new Error('`protocol` must be HTTP, HTTPS, or #{protocol}.');
    }
    if (!redirectResponse.statusCode || !/^HTTP_30[12]$/.test(redirectResponse.statusCode)) {
        throw new Error('`statusCode` must be HTTP_301 or HTTP_302.');
    }
}
exports.validateRedirectResponse = validateRedirectResponse;
//# sourceMappingURL=data:application/json;base64,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