"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationListener = void 0;
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const base_listener_1 = require("../shared/base-listener");
const enums_1 = require("../shared/enums");
const listener_certificate_1 = require("../shared/listener-certificate");
const util_1 = require("../shared/util");
const application_listener_action_1 = require("./application-listener-action");
const application_listener_certificate_1 = require("./application-listener-certificate");
const application_listener_rule_1 = require("./application-listener-rule");
const application_target_group_1 = require("./application-target-group");
/**
 * Define an ApplicationListener
 *
 * @resource AWS::ElasticLoadBalancingV2::Listener
 */
class ApplicationListener extends base_listener_1.BaseListener {
    constructor(scope, id, props) {
        const [protocol, port] = util_1.determineProtocolAndPort(props.protocol, props.port);
        if (protocol === undefined || port === undefined) {
            throw new Error('At least one of \'port\' or \'protocol\' is required');
        }
        super(scope, id, {
            loadBalancerArn: props.loadBalancer.loadBalancerArn,
            certificates: core_1.Lazy.anyValue({ produce: () => this.certificateArns.map(certificateArn => ({ certificateArn })) }, { omitEmptyArray: true }),
            protocol,
            port,
            sslPolicy: props.sslPolicy,
        });
        this.loadBalancer = props.loadBalancer;
        this.protocol = protocol;
        this.certificateArns = [];
        // Attach certificates
        if (props.certificateArns && props.certificateArns.length > 0) {
            this.addCertificateArns('ListenerCertificate', props.certificateArns);
        }
        if (props.certificates && props.certificates.length > 0) {
            this.addCertificates('DefaultCertificates', props.certificates);
        }
        // This listener edits the securitygroup of the load balancer,
        // but adds its own default port.
        this.connections = new ec2.Connections({
            securityGroups: props.loadBalancer.connections.securityGroups,
            defaultPort: ec2.Port.tcp(port),
        });
        if (props.defaultAction && props.defaultTargetGroups) {
            throw new Error('Specify at most one of \'defaultAction\' and \'defaultTargetGroups\'');
        }
        if (props.defaultAction) {
            this.setDefaultAction(props.defaultAction);
        }
        if (props.defaultTargetGroups) {
            this.setDefaultAction(application_listener_action_1.ListenerAction.forward(props.defaultTargetGroups));
        }
        if (props.open !== false) {
            this.connections.allowDefaultPortFrom(ec2.Peer.anyIpv4(), `Allow from anyone on port ${port}`);
            if (this.loadBalancer.ipAddressType === enums_1.IpAddressType.DUAL_STACK) {
                this.connections.allowDefaultPortFrom(ec2.Peer.anyIpv6(), `Allow from anyone on port ${port}`);
            }
        }
    }
    /**
     * Import an existing listener
     */
    static fromApplicationListenerAttributes(scope, id, attrs) {
        return new ImportedApplicationListener(scope, id, attrs);
    }
    /**
     * Add one or more certificates to this listener.
     *
     * After the first certificate, this creates ApplicationListenerCertificates
     * resources since cloudformation requires the certificates array on the
     * listener resource to have a length of 1.
     *
     * @deprecated Use `addCertificates` instead.
     */
    addCertificateArns(id, arns) {
        this.addCertificates(id, arns.map(listener_certificate_1.ListenerCertificate.fromArn));
    }
    /**
     * Add one or more certificates to this listener.
     *
     * After the first certificate, this creates ApplicationListenerCertificates
     * resources since cloudformation requires the certificates array on the
     * listener resource to have a length of 1.
     */
    addCertificates(id, certificates) {
        const additionalCerts = [...certificates];
        if (this.certificateArns.length === 0 && additionalCerts.length > 0) {
            const first = additionalCerts.splice(0, 1)[0];
            this.certificateArns.push(first.certificateArn);
        }
        if (additionalCerts.length > 0) {
            new application_listener_certificate_1.ApplicationListenerCertificate(this, id, {
                listener: this,
                certificates: additionalCerts,
            });
        }
    }
    /**
     * Perform the given default action on incoming requests
     *
     * This allows full control of the default action of the load balancer,
     * including Action chaining, fixed responses and redirect responses. See
     * the `ListenerAction` class for all options.
     *
     * It's possible to add routing conditions to the Action added in this way.
     * At least one Action must be added without conditions (which becomes the
     * default Action).
     */
    addAction(id, props) {
        checkAddRuleProps(props);
        if (props.priority !== undefined) {
            // New rule
            //
            // TargetGroup.registerListener is called inside ApplicationListenerRule.
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                conditions: props.conditions,
                hostHeader: props.hostHeader,
                pathPattern: props.pathPattern,
                pathPatterns: props.pathPatterns,
                priority: props.priority,
                action: props.action,
            });
        }
        else {
            // New default target with these targetgroups
            this.setDefaultAction(props.action);
        }
    }
    /**
     * Load balance incoming requests to the given target groups.
     *
     * All target groups will be load balanced to with equal weight and without
     * stickiness. For a more complex configuration than that, use `addAction()`.
     *
     * It's possible to add routing conditions to the TargetGroups added in this
     * way. At least one TargetGroup must be added without conditions (which will
     * become the default Action for this listener).
     */
    addTargetGroups(id, props) {
        checkAddRuleProps(props);
        if (props.priority !== undefined) {
            // New rule
            //
            // TargetGroup.registerListener is called inside ApplicationListenerRule.
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                conditions: props.conditions,
                hostHeader: props.hostHeader,
                pathPattern: props.pathPattern,
                pathPatterns: props.pathPatterns,
                priority: props.priority,
                targetGroups: props.targetGroups,
            });
        }
        else {
            // New default target with these targetgroups
            this.setDefaultAction(application_listener_action_1.ListenerAction.forward(props.targetGroups));
        }
    }
    /**
     * Load balance incoming requests to the given load balancing targets.
     *
     * This method implicitly creates an ApplicationTargetGroup for the targets
     * involved, and a 'forward' action to route traffic to the given TargetGroup.
     *
     * If you want more control over the precise setup, create the TargetGroup
     * and use `addAction` yourself.
     *
     * It's possible to add conditions to the targets added in this way. At least
     * one set of targets must be added without conditions.
     *
     * @returns The newly created target group
     */
    addTargets(id, props) {
        if (!this.loadBalancer.vpc) {
            // eslint-disable-next-line max-len
            throw new Error('Can only call addTargets() when using a constructed Load Balancer or an imported Load Balancer with specified vpc; construct a new TargetGroup and use addTargetGroup');
        }
        const group = new application_target_group_1.ApplicationTargetGroup(this, id + 'Group', {
            deregistrationDelay: props.deregistrationDelay,
            healthCheck: props.healthCheck,
            port: props.port,
            protocol: props.protocol,
            slowStart: props.slowStart,
            stickinessCookieDuration: props.stickinessCookieDuration,
            targetGroupName: props.targetGroupName,
            targets: props.targets,
            vpc: this.loadBalancer.vpc,
        });
        this.addTargetGroups(id, {
            conditions: props.conditions,
            hostHeader: props.hostHeader,
            pathPattern: props.pathPattern,
            pathPatterns: props.pathPatterns,
            priority: props.priority,
            targetGroups: [group],
        });
        return group;
    }
    /**
     * Add a fixed response
     *
     * @deprecated Use `addAction()` instead
     */
    addFixedResponse(id, props) {
        checkAddRuleProps(props);
        const fixedResponse = {
            statusCode: props.statusCode,
            contentType: props.contentType,
            messageBody: props.messageBody,
        };
        application_listener_rule_1.validateFixedResponse(fixedResponse);
        if (props.priority) {
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                priority: props.priority,
                fixedResponse,
                ...props,
            });
        }
        else {
            this.setDefaultAction(application_listener_action_1.ListenerAction.fixedResponse(core_1.Token.asNumber(props.statusCode), {
                contentType: props.contentType,
                messageBody: props.messageBody,
            }));
        }
    }
    /**
     * Add a redirect response
     *
     * @deprecated Use `addAction()` instead
     */
    addRedirectResponse(id, props) {
        checkAddRuleProps(props);
        const redirectResponse = {
            host: props.host,
            path: props.path,
            port: props.port,
            protocol: props.protocol,
            query: props.query,
            statusCode: props.statusCode,
        };
        application_listener_rule_1.validateRedirectResponse(redirectResponse);
        if (props.priority) {
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                priority: props.priority,
                redirectResponse,
                ...props,
            });
        }
        else {
            this.setDefaultAction(application_listener_action_1.ListenerAction.redirect({
                host: props.host,
                path: props.path,
                port: props.port,
                protocol: props.protocol,
                query: props.query,
                permanent: props.statusCode === 'HTTP_301',
            }));
        }
    }
    /**
     * Register that a connectable that has been added to this load balancer.
     *
     * Don't call this directly. It is called by ApplicationTargetGroup.
     */
    registerConnectable(connectable, portRange) {
        connectable.connections.allowFrom(this.loadBalancer, portRange, 'Load balancer to target');
    }
    /**
     * Validate this listener.
     */
    validate() {
        const errors = super.validate();
        if (this.protocol === enums_1.ApplicationProtocol.HTTPS && this.certificateArns.length === 0) {
            errors.push('HTTPS Listener needs at least one certificate (call addCertificates)');
        }
        return errors;
    }
    /**
     * Wrapper for _setDefaultAction which does a type-safe bind
     */
    setDefaultAction(action) {
        action.bind(this, this);
        this._setDefaultAction(action);
    }
}
exports.ApplicationListener = ApplicationListener;
class ImportedApplicationListener extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.listenerArn = props.listenerArn;
        const defaultPort = props.defaultPort !== undefined ? ec2.Port.tcp(props.defaultPort) : undefined;
        let securityGroup;
        if (props.securityGroup) {
            securityGroup = props.securityGroup;
        }
        else if (props.securityGroupId) {
            securityGroup = ec2.SecurityGroup.fromSecurityGroupId(scope, 'SecurityGroup', props.securityGroupId, {
                allowAllOutbound: props.securityGroupAllowsAllOutbound,
            });
        }
        else {
            throw new Error('Either `securityGroup` or `securityGroupId` must be specified to import an application listener.');
        }
        this.connections = new ec2.Connections({
            securityGroups: [securityGroup],
            defaultPort,
        });
    }
    /**
     * Add one or more certificates to this listener.
     */
    addCertificateArns(id, arns) {
        new application_listener_certificate_1.ApplicationListenerCertificate(this, id, {
            listener: this,
            certificateArns: arns,
        });
    }
    /**
     * Load balance incoming requests to the given target groups.
     *
     * It's possible to add conditions to the TargetGroups added in this way.
     * At least one TargetGroup must be added without conditions.
     */
    addTargetGroups(id, props) {
        checkAddRuleProps(props);
        if (props.priority !== undefined) {
            // New rule
            new application_listener_rule_1.ApplicationListenerRule(this, id, {
                listener: this,
                conditions: props.conditions,
                hostHeader: props.hostHeader,
                pathPattern: props.pathPattern,
                pathPatterns: props.pathPatterns,
                priority: props.priority,
                targetGroups: props.targetGroups,
            });
        }
        else {
            throw new Error('Cannot add default Target Groups to imported ApplicationListener');
        }
    }
    /**
     * Load balance incoming requests to the given load balancing targets.
     *
     * This method implicitly creates an ApplicationTargetGroup for the targets
     * involved.
     *
     * It's possible to add conditions to the targets added in this way. At least
     * one set of targets must be added without conditions.
     *
     * @returns The newly created target group
     */
    addTargets(_id, _props) {
        // eslint-disable-next-line max-len
        throw new Error('Can only call addTargets() when using a constructed ApplicationListener; construct a new TargetGroup and use addTargetGroup.');
    }
    /**
     * Register that a connectable that has been added to this load balancer.
     *
     * Don't call this directly. It is called by ApplicationTargetGroup.
     */
    registerConnectable(connectable, portRange) {
        this.connections.allowTo(connectable, portRange, 'Load balancer to target');
    }
}
function checkAddRuleProps(props) {
    var _a;
    const conditionsCount = ((_a = props.conditions) === null || _a === void 0 ? void 0 : _a.length) || 0;
    const hasAnyConditions = conditionsCount !== 0 ||
        props.hostHeader !== undefined || props.pathPattern !== undefined || props.pathPatterns !== undefined;
    const hasPriority = props.priority !== undefined;
    if (hasAnyConditions !== hasPriority) {
        throw new Error('Setting \'conditions\', \'pathPattern\' or \'hostHeader\' also requires \'priority\', and vice versa');
    }
}
//# sourceMappingURL=data:application/json;base64,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