"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Domain = exports.TLSSecurityPolicy = exports.ElasticsearchVersion = void 0;
const url_1 = require("url");
const aws_cloudwatch_1 = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const logs = require("../../aws-logs"); // Automatically re-written from '@aws-cdk/aws-logs'
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const elasticsearch_access_policy_1 = require("./elasticsearch-access-policy");
const elasticsearch_generated_1 = require("./elasticsearch.generated");
const log_group_resource_policy_1 = require("./log-group-resource-policy");
const perms = require("./perms");
/**
 * Elasticsearch version
 */
class ElasticsearchVersion {
    /**
     *
     * @param version Elasticsearch version number
     */
    constructor(version) {
        this.version = version;
    }
    /**
     * Custom Elasticsearch version
     * @param version custom version number
     */
    static of(version) { return new ElasticsearchVersion(version); }
}
exports.ElasticsearchVersion = ElasticsearchVersion;
/** AWS Elasticsearch 1.5 */
ElasticsearchVersion.V1_5 = ElasticsearchVersion.of('1.5');
/** AWS Elasticsearch 2.3 */
ElasticsearchVersion.V2_3 = ElasticsearchVersion.of('2.3');
/** AWS Elasticsearch 5.1 */
ElasticsearchVersion.V5_1 = ElasticsearchVersion.of('5.1');
/** AWS Elasticsearch 5.3 */
ElasticsearchVersion.V5_3 = ElasticsearchVersion.of('5.3');
/** AWS Elasticsearch 5.5 */
ElasticsearchVersion.V5_5 = ElasticsearchVersion.of('5.5');
/** AWS Elasticsearch 5.6 */
ElasticsearchVersion.V5_6 = ElasticsearchVersion.of('5.6');
/** AWS Elasticsearch 6.0 */
ElasticsearchVersion.V6_0 = ElasticsearchVersion.of('6.0');
/** AWS Elasticsearch 6.2 */
ElasticsearchVersion.V6_2 = ElasticsearchVersion.of('6.2');
/** AWS Elasticsearch 6.3 */
ElasticsearchVersion.V6_3 = ElasticsearchVersion.of('6.3');
/** AWS Elasticsearch 6.4 */
ElasticsearchVersion.V6_4 = ElasticsearchVersion.of('6.4');
/** AWS Elasticsearch 6.5 */
ElasticsearchVersion.V6_5 = ElasticsearchVersion.of('6.5');
/** AWS Elasticsearch 6.7 */
ElasticsearchVersion.V6_7 = ElasticsearchVersion.of('6.7');
/** AWS Elasticsearch 6.8 */
ElasticsearchVersion.V6_8 = ElasticsearchVersion.of('6.8');
/** AWS Elasticsearch 7.1 */
ElasticsearchVersion.V7_1 = ElasticsearchVersion.of('7.1');
/** AWS Elasticsearch 7.4 */
ElasticsearchVersion.V7_4 = ElasticsearchVersion.of('7.4');
/** AWS Elasticsearch 7.7 */
ElasticsearchVersion.V7_7 = ElasticsearchVersion.of('7.7');
/**
 * The minimum TLS version required for traffic to the domain.
 */
var TLSSecurityPolicy;
(function (TLSSecurityPolicy) {
    /** Cipher suite TLS 1.0 */
    TLSSecurityPolicy["TLS_1_0"] = "Policy-Min-TLS-1-0-2019-07";
    /** Cipher suite TLS 1.2 */
    TLSSecurityPolicy["TLS_1_2"] = "Policy-Min-TLS-1-2-2019-07";
})(TLSSecurityPolicy = exports.TLSSecurityPolicy || (exports.TLSSecurityPolicy = {}));
/**
 * A new or imported domain.
 */
class DomainBase extends cdk.Resource {
    /**
     * Grant read permissions for this domain and its contents to an IAM
     * principal (Role/Group/User).
     *
     * @param identity The principal
     */
    grantRead(identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * Grant write permissions for this domain and its contents to an IAM
     * principal (Role/Group/User).
     *
     * @param identity The principal
     */
    grantWrite(identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * Grant read/write permissions for this domain and its contents to an IAM
     * principal (Role/Group/User).
     *
     * @param identity The principal
     */
    grantReadWrite(identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * Grant read permissions for an index in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param index The index to grant permissions for
     * @param identity The principal
     */
    grantIndexRead(index, identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * Grant write permissions for an index in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param index The index to grant permissions for
     * @param identity The principal
     */
    grantIndexWrite(index, identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * Grant read/write permissions for an index in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param index The index to grant permissions for
     * @param identity The principal
     */
    grantIndexReadWrite(index, identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * Grant read permissions for a specific path in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param path The path to grant permissions for
     * @param identity The principal
     */
    grantPathRead(path, identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * Grant write permissions for a specific path in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param path The path to grant permissions for
     * @param identity The principal
     */
    grantPathWrite(path, identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * Grant read/write permissions for a specific path in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param path The path to grant permissions for
     * @param identity The principal
     */
    grantPathReadWrite(path, identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * Return the given named metric for this Domain.
     */
    metric(metricName, props) {
        return new aws_cloudwatch_1.Metric({
            namespace: 'AWS/ES',
            metricName,
            dimensions: {
                DomainName: this.domainName,
                ClientId: this.stack.account,
            },
            ...props,
        });
    }
    /**
     * Metric for the time the cluster status is red.
     *
     * @default maximum over 5 minutes
     */
    metricClusterStatusRed(props) {
        return this.metric('ClusterStatus.red', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for the time the cluster status is yellow.
     *
     * @default maximum over 5 minutes
     */
    metricClusterStatusYellow(props) {
        return this.metric('ClusterStatus.yellow', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for the storage space of nodes in the cluster.
     *
     * @default minimum over 5 minutes
     */
    metricFreeStorageSpace(props) {
        return this.metric('FreeStorageSpace', {
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...props,
        });
    }
    /**
     * Metric for the cluster blocking index writes.
     *
     * @default maximum over 1 minute
     */
    metricClusterIndexWriteBlocked(props) {
        return this.metric('ClusterIndexWriteBlocked', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            period: cdk.Duration.minutes(1),
            ...props,
        });
    }
    /**
     * Metric for the number of nodes.
     *
     * @default minimum over 1 hour
     */
    metricNodes(props) {
        return this.metric('Nodes', {
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            period: cdk.Duration.hours(1),
            ...props,
        });
    }
    /**
     * Metric for automated snapshot failures.
     *
     * @default maximum over 5 minutes
     */
    metricAutomatedSnapshotFailure(props) {
        return this.metric('AutomatedSnapshotFailure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for CPU utilization.
     *
     * @default maximum over 5 minutes
     */
    metricCPUUtilization(props) {
        return this.metric('CPUUtilization', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for JVM memory pressure.
     *
     * @default maximum over 5 minutes
     */
    metricJVMMemoryPressure(props) {
        return this.metric('JVMMemoryPressure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for master CPU utilization.
     *
     * @default maximum over 5 minutes
     */
    metricMasterCPUUtilization(props) {
        return this.metric('MasterCPUUtilization', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for master JVM memory pressure.
     *
     * @default maximum over 5 minutes
     */
    metricMasterJVMMemoryPressure(props) {
        return this.metric('MasterJVMMemoryPressure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for KMS key errors.
     *
     * @default maximum over 5 minutes
     */
    metricKMSKeyError(props) {
        return this.metric('KMSKeyError', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for KMS key being inaccessible.
     *
     * @default maximum over 5 minutes
     */
    metricKMSKeyInaccessible(props) {
        return this.metric('KMSKeyInaccessible', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for number of searchable documents.
     *
     * @default maximum over 5 minutes
     */
    metricSearchableDocuments(props) {
        return this.metric('SearchableDocuments', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for search latency.
     *
     * @default p99 over 5 minutes
     */
    metricSearchLatency(props) {
        return this.metric('SearchLatencyP99', { statistic: 'p99', ...props });
    }
    /**
     * Metric for indexing latency.
     *
     * @default p99 over 5 minutes
     */
    metricIndexingLatency(props) {
        return this.metric('IndexingLatencyP99', { statistic: 'p99', ...props });
    }
    grant(grantee, domainActions, resourceArn, ...otherResourceArns) {
        const resourceArns = [resourceArn, ...otherResourceArns];
        const grant = iam.Grant.addToPrincipal({
            grantee,
            actions: domainActions,
            resourceArns,
            scope: this,
        });
        return grant;
    }
}
/**
 * Provides an Elasticsearch domain.
 */
class Domain extends DomainBase {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0, _1, _2, _3, _4, _5, _6, _7, _8, _9, _10, _11, _12, _13, _14, _15, _16, _17, _18, _19, _20, _21, _22, _23, _24, _25, _26, _27, _28, _29;
        super(scope, id, {
            physicalName: props.domainName,
        });
        const defaultInstanceType = 'r5.large.elasticsearch';
        const dedicatedMasterType = (_c = (_b = (_a = props.capacity) === null || _a === void 0 ? void 0 : _a.masterNodeInstanceType) === null || _b === void 0 ? void 0 : _b.toLowerCase()) !== null && _c !== void 0 ? _c : defaultInstanceType;
        const dedicatedMasterCount = (_e = (_d = props.capacity) === null || _d === void 0 ? void 0 : _d.masterNodes) !== null && _e !== void 0 ? _e : 0;
        const dedicatedMasterEnabled = dedicatedMasterCount > 0;
        const instanceType = (_h = (_g = (_f = props.capacity) === null || _f === void 0 ? void 0 : _f.dataNodeInstanceType) === null || _g === void 0 ? void 0 : _g.toLowerCase()) !== null && _h !== void 0 ? _h : defaultInstanceType;
        const instanceCount = (_k = (_j = props.capacity) === null || _j === void 0 ? void 0 : _j.dataNodes) !== null && _k !== void 0 ? _k : 1;
        const availabilityZoneCount = (_m = (_l = props.zoneAwareness) === null || _l === void 0 ? void 0 : _l.availabilityZoneCount) !== null && _m !== void 0 ? _m : 2;
        if (![2, 3].includes(availabilityZoneCount)) {
            throw new Error('Invalid zone awareness configuration; availabilityZoneCount must be 2 or 3');
        }
        const zoneAwarenessEnabled = (_p = (_o = props.zoneAwareness) === null || _o === void 0 ? void 0 : _o.enabled) !== null && _p !== void 0 ? _p : ((_q = props.zoneAwareness) === null || _q === void 0 ? void 0 : _q.availabilityZoneCount) != null;
        // If VPC options are supplied ensure that the number of subnets matches the number AZ
        if (props.vpcOptions != null && zoneAwarenessEnabled &&
            new Set((_r = props.vpcOptions) === null || _r === void 0 ? void 0 : _r.subnets.map((subnet) => subnet.availabilityZone)).size < availabilityZoneCount) {
            throw new Error('When providing vpc options you need to provide a subnet for each AZ you are using');
        }
        ;
        if ([dedicatedMasterType, instanceType].some(t => !t.endsWith('.elasticsearch'))) {
            throw new Error('Master and data node instance types must end with ".elasticsearch".');
        }
        const elasticsearchVersion = props.version.version;
        const elasticsearchVersionNum = parseVersion(props.version);
        if (elasticsearchVersionNum <= 7.7 &&
            ![
                1.5, 2.3, 5.1, 5.3, 5.5, 5.6, 6.0,
                6.2, 6.3, 6.4, 6.5, 6.7, 6.8, 7.1, 7.4,
                7.7,
            ].includes(elasticsearchVersionNum)) {
            throw new Error(`Unknown Elasticsearch version: ${elasticsearchVersion}`);
        }
        const unsignedBasicAuthEnabled = (_s = props.useUnsignedBasicAuth) !== null && _s !== void 0 ? _s : false;
        if (unsignedBasicAuthEnabled) {
            if (props.enforceHttps == false) {
                throw new Error('You cannot disable HTTPS and use unsigned basic auth');
            }
            if (props.nodeToNodeEncryption == false) {
                throw new Error('You cannot disable node to node encryption and use unsigned basic auth');
            }
            if (((_t = props.encryptionAtRest) === null || _t === void 0 ? void 0 : _t.enabled) == false) {
                throw new Error('You cannot disable encryption at rest and use unsigned basic auth');
            }
        }
        const unsignedAccessPolicy = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['es:ESHttp*'],
            principals: [new iam.Anyone()],
            resources: [cdk.Lazy.stringValue({ produce: () => `${this.domainArn}/*` })],
        });
        const masterUserArn = (_u = props.fineGrainedAccessControl) === null || _u === void 0 ? void 0 : _u.masterUserArn;
        const masterUserNameProps = (_v = props.fineGrainedAccessControl) === null || _v === void 0 ? void 0 : _v.masterUserName;
        // If basic auth is enabled set the user name to admin if no other user info is supplied.
        const masterUserName = unsignedBasicAuthEnabled
            ? (masterUserArn == null ? (masterUserNameProps !== null && masterUserNameProps !== void 0 ? masterUserNameProps : 'admin') : undefined)
            : masterUserNameProps;
        if (masterUserArn != null && masterUserName != null) {
            throw new Error('Invalid fine grained access control settings. Only provide one of master user ARN or master user name. Not both.');
        }
        const advancedSecurityEnabled = (masterUserArn !== null && masterUserArn !== void 0 ? masterUserArn : masterUserName) != null;
        const internalUserDatabaseEnabled = masterUserName != null;
        const masterUserPasswordProp = (_w = props.fineGrainedAccessControl) === null || _w === void 0 ? void 0 : _w.masterUserPassword;
        const createMasterUserPassword = () => {
            return new secretsmanager.Secret(this, 'MasterUser', {
                generateSecretString: {
                    secretStringTemplate: JSON.stringify({
                        username: masterUserName,
                    }),
                    generateStringKey: 'password',
                    excludeCharacters: "{}'\\*[]()`",
                },
            })
                .secretValueFromJson('password');
        };
        this.masterUserPassword = internalUserDatabaseEnabled ?
            (masterUserPasswordProp !== null && masterUserPasswordProp !== void 0 ? masterUserPasswordProp : createMasterUserPassword())
            : undefined;
        const encryptionAtRestEnabled = (_y = (_x = props.encryptionAtRest) === null || _x === void 0 ? void 0 : _x.enabled) !== null && _y !== void 0 ? _y : (((_z = props.encryptionAtRest) === null || _z === void 0 ? void 0 : _z.kmsKey) != null || unsignedBasicAuthEnabled);
        const nodeToNodeEncryptionEnabled = (_0 = props.nodeToNodeEncryption) !== null && _0 !== void 0 ? _0 : unsignedBasicAuthEnabled;
        const volumeSize = (_2 = (_1 = props.ebs) === null || _1 === void 0 ? void 0 : _1.volumeSize) !== null && _2 !== void 0 ? _2 : 10;
        const volumeType = (_4 = (_3 = props.ebs) === null || _3 === void 0 ? void 0 : _3.volumeType) !== null && _4 !== void 0 ? _4 : ec2.EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
        const ebsEnabled = (_6 = (_5 = props.ebs) === null || _5 === void 0 ? void 0 : _5.enabled) !== null && _6 !== void 0 ? _6 : true;
        const enforceHttps = (_7 = props.enforceHttps) !== null && _7 !== void 0 ? _7 : unsignedBasicAuthEnabled;
        function isInstanceType(t) {
            return dedicatedMasterType.startsWith(t) || instanceType.startsWith(t);
        }
        ;
        function isSomeInstanceType(...instanceTypes) {
            return instanceTypes.some(isInstanceType);
        }
        ;
        function isEveryInstanceType(...instanceTypes) {
            return instanceTypes.some(t => dedicatedMasterType.startsWith(t))
                && instanceTypes.some(t => instanceType.startsWith(t));
        }
        ;
        // Validate feature support for the given Elasticsearch version, per
        // https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/aes-features-by-version.html
        if (elasticsearchVersionNum < 5.1) {
            if (((_8 = props.logging) === null || _8 === void 0 ? void 0 : _8.slowIndexLogEnabled) || ((_9 = props.logging) === null || _9 === void 0 ? void 0 : _9.appLogEnabled)
                || ((_10 = props.logging) === null || _10 === void 0 ? void 0 : _10.slowSearchLogEnabled)) {
                throw new Error('Error and slow logs publishing requires Elasticsearch version 5.1 or later.');
            }
            if ((_11 = props.encryptionAtRest) === null || _11 === void 0 ? void 0 : _11.enabled) {
                throw new Error('Encryption of data at rest requires Elasticsearch version 5.1 or later.');
            }
            if (props.cognitoKibanaAuth != null) {
                throw new Error('Cognito authentication for Kibana requires Elasticsearch version 5.1 or later.');
            }
            if (isSomeInstanceType('c5', 'i3', 'm5', 'r5')) {
                throw new Error('C5, I3, M5, and R5 instance types require Elasticsearch version 5.1 or later.');
            }
        }
        if (elasticsearchVersionNum < 6.0) {
            if (props.nodeToNodeEncryption) {
                throw new Error('Node-to-node encryption requires Elasticsearch version 6.0 or later.');
            }
        }
        if (elasticsearchVersionNum < 6.7) {
            if (unsignedBasicAuthEnabled) {
                throw new Error('Using unsigned basic auth requires Elasticsearch version 6.7 or later.');
            }
            if (advancedSecurityEnabled) {
                throw new Error('Fine-grained access control requires Elasticsearch version 6.7 or later.');
            }
        }
        // Validate against instance type restrictions, per
        // https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/aes-supported-instance-types.html
        if (isInstanceType('i3') && ebsEnabled) {
            throw new Error('I3 instance types do not support EBS storage volumes.');
        }
        if (isSomeInstanceType('m3', 'r3', 't2') && encryptionAtRestEnabled) {
            throw new Error('M3, R3, and T2 instance types do not support encryption of data at rest.');
        }
        if (isInstanceType('t2.micro') && elasticsearchVersionNum > 2.3) {
            throw new Error('The t2.micro.elasticsearch instance type supports only Elasticsearch 1.5 and 2.3.');
        }
        // Only R3 and I3 support instance storage, per
        // https://aws.amazon.com/elasticsearch-service/pricing/
        if (!ebsEnabled && !isEveryInstanceType('r3', 'i3')) {
            throw new Error('EBS volumes are required when using instance types other than r3 or i3.');
        }
        // Fine-grained access control requires node-to-node encryption, encryption at rest,
        // and enforced HTTPS.
        if (advancedSecurityEnabled) {
            if (!nodeToNodeEncryptionEnabled) {
                throw new Error('Node-to-node encryption is required when fine-grained access control is enabled.');
            }
            if (!encryptionAtRestEnabled) {
                throw new Error('Encryption-at-rest is required when fine-grained access control is enabled.');
            }
            if (!enforceHttps) {
                throw new Error('Enforce HTTPS is required when fine-grained access control is enabled.');
            }
        }
        let cfnVpcOptions;
        if (props.vpcOptions) {
            cfnVpcOptions = {
                securityGroupIds: props.vpcOptions.securityGroups.map((sg) => sg.securityGroupId),
                subnetIds: props.vpcOptions.subnets.map((subnet) => subnet.subnetId),
            };
        }
        // Setup logging
        const logGroups = [];
        if ((_12 = props.logging) === null || _12 === void 0 ? void 0 : _12.slowSearchLogEnabled) {
            this.slowSearchLogGroup = (_13 = props.logging.slowSearchLogGroup) !== null && _13 !== void 0 ? _13 : new logs.LogGroup(scope, 'SlowSearchLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.slowSearchLogGroup);
        }
        ;
        if ((_14 = props.logging) === null || _14 === void 0 ? void 0 : _14.slowIndexLogEnabled) {
            this.slowIndexLogGroup = (_15 = props.logging.slowIndexLogGroup) !== null && _15 !== void 0 ? _15 : new logs.LogGroup(scope, 'SlowIndexLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.slowIndexLogGroup);
        }
        ;
        if ((_16 = props.logging) === null || _16 === void 0 ? void 0 : _16.appLogEnabled) {
            this.appLogGroup = (_17 = props.logging.appLogGroup) !== null && _17 !== void 0 ? _17 : new logs.LogGroup(scope, 'AppLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.appLogGroup);
        }
        ;
        let logGroupResourcePolicy = null;
        if (logGroups.length > 0) {
            const logPolicyStatement = new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['logs:PutLogEvents', 'logs:CreateLogStream'],
                resources: logGroups.map((lg) => lg.logGroupArn),
                principals: [new iam.ServicePrincipal('es.amazonaws.com')],
            });
            // Use a custom resource to set the log group resource policy since it is not supported by CDK and cfn.
            // https://github.com/aws/aws-cdk/issues/5343
            logGroupResourcePolicy = new log_group_resource_policy_1.LogGroupResourcePolicy(this, 'ESLogGroupPolicy', {
                policyName: 'ESLogPolicy',
                policyStatements: [logPolicyStatement],
            });
        }
        // Create the domain
        this.domain = new elasticsearch_generated_1.CfnDomain(this, 'Resource', {
            domainName: this.physicalName,
            elasticsearchVersion,
            elasticsearchClusterConfig: {
                dedicatedMasterEnabled,
                dedicatedMasterCount: dedicatedMasterEnabled
                    ? dedicatedMasterCount
                    : undefined,
                dedicatedMasterType: dedicatedMasterEnabled
                    ? dedicatedMasterType
                    : undefined,
                instanceCount,
                instanceType,
                zoneAwarenessEnabled,
                zoneAwarenessConfig: zoneAwarenessEnabled
                    ? { availabilityZoneCount }
                    : undefined,
            },
            ebsOptions: {
                ebsEnabled,
                volumeSize: ebsEnabled ? volumeSize : undefined,
                volumeType: ebsEnabled ? volumeType : undefined,
                iops: ebsEnabled ? (_18 = props.ebs) === null || _18 === void 0 ? void 0 : _18.iops : undefined,
            },
            encryptionAtRestOptions: {
                enabled: encryptionAtRestEnabled,
                kmsKeyId: encryptionAtRestEnabled
                    ? (_20 = (_19 = props.encryptionAtRest) === null || _19 === void 0 ? void 0 : _19.kmsKey) === null || _20 === void 0 ? void 0 : _20.keyId : undefined,
            },
            nodeToNodeEncryptionOptions: { enabled: nodeToNodeEncryptionEnabled },
            logPublishingOptions: {
                ES_APPLICATION_LOGS: {
                    enabled: this.appLogGroup != null,
                    cloudWatchLogsLogGroupArn: (_21 = this.appLogGroup) === null || _21 === void 0 ? void 0 : _21.logGroupArn,
                },
                SEARCH_SLOW_LOGS: {
                    enabled: this.slowSearchLogGroup != null,
                    cloudWatchLogsLogGroupArn: (_22 = this.slowSearchLogGroup) === null || _22 === void 0 ? void 0 : _22.logGroupArn,
                },
                INDEX_SLOW_LOGS: {
                    enabled: this.slowIndexLogGroup != null,
                    cloudWatchLogsLogGroupArn: (_23 = this.slowIndexLogGroup) === null || _23 === void 0 ? void 0 : _23.logGroupArn,
                },
            },
            cognitoOptions: {
                enabled: props.cognitoKibanaAuth != null,
                identityPoolId: (_24 = props.cognitoKibanaAuth) === null || _24 === void 0 ? void 0 : _24.identityPoolId,
                roleArn: (_25 = props.cognitoKibanaAuth) === null || _25 === void 0 ? void 0 : _25.role.roleArn,
                userPoolId: (_26 = props.cognitoKibanaAuth) === null || _26 === void 0 ? void 0 : _26.userPoolId,
            },
            vpcOptions: cfnVpcOptions,
            snapshotOptions: props.automatedSnapshotStartHour
                ? { automatedSnapshotStartHour: props.automatedSnapshotStartHour }
                : undefined,
            domainEndpointOptions: {
                enforceHttps,
                tlsSecurityPolicy: (_27 = props.tlsSecurityPolicy) !== null && _27 !== void 0 ? _27 : TLSSecurityPolicy.TLS_1_0,
            },
            advancedSecurityOptions: advancedSecurityEnabled
                ? {
                    enabled: true,
                    internalUserDatabaseEnabled,
                    masterUserOptions: {
                        masterUserArn: masterUserArn,
                        masterUserName: masterUserName,
                        masterUserPassword: (_28 = this.masterUserPassword) === null || _28 === void 0 ? void 0 : _28.toString(),
                    },
                }
                : undefined,
        });
        if (logGroupResourcePolicy) {
            this.domain.node.addDependency(logGroupResourcePolicy);
        }
        if (props.domainName) {
            this.node.addMetadata('aws:cdk:hasPhysicalName', props.domainName);
        }
        this.domainName = this.getResourceNameAttribute(this.domain.ref);
        this.domainEndpoint = this.domain.getAtt('DomainEndpoint').toString();
        this.domainArn = this.getResourceArnAttribute(this.domain.attrArn, {
            service: 'es',
            resource: 'domain',
            resourceName: this.physicalName,
        });
        const accessPolicyStatements = unsignedBasicAuthEnabled
            ? ((_29 = props.accessPolicies) !== null && _29 !== void 0 ? _29 : []).concat(unsignedAccessPolicy)
            : props.accessPolicies;
        if (accessPolicyStatements != null) {
            const accessPolicy = new elasticsearch_access_policy_1.ElasticsearchAccessPolicy(this, 'ESAccessPolicy', {
                domainName: this.domainName,
                domainArn: this.domainArn,
                accessPolicies: accessPolicyStatements,
            });
            accessPolicy.node.addDependency(this.domain);
        }
    }
    /**
     * Creates a Domain construct that represents an external domain via domain endpoint.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param domainEndpoint The domain's endpoint.
     */
    static fromDomainEndpoint(scope, id, domainEndpoint) {
        const stack = cdk.Stack.of(scope);
        const domainName = extractNameFromEndpoint(domainEndpoint);
        const domainArn = stack.formatArn({
            service: 'es',
            resource: 'domain',
            resourceName: domainName,
        });
        return Domain.fromDomainAttributes(scope, id, {
            domainArn,
            domainEndpoint,
        });
    }
    /**
     * Creates a Domain construct that represents an external domain.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `DomainAttributes` object.
     */
    static fromDomainAttributes(scope, id, attrs) {
        const { domainArn, domainEndpoint } = attrs;
        const domainName = extractNameFromEndpoint(domainEndpoint);
        return new class extends DomainBase {
            constructor() {
                super(scope, id);
                this.domainArn = domainArn;
                this.domainName = domainName;
                this.domainEndpoint = domainEndpoint;
            }
        };
    }
}
exports.Domain = Domain;
/**
 * Given an Elasticsearch domain endpoint, returns a CloudFormation expression that
 * extracts the domain name.
 *
 * Domain endpoints look like this:
 *
 *   https://example-domain-jcjotrt6f7otem4sqcwbch3c4u.us-east-1.es.amazonaws.com
 *   https://<domain-name>-<suffix>.<region>.es.amazonaws.com
 *
 * ..which means that in order to extract the domain name from the endpoint, we can
 * split the endpoint using "-<suffix>" and select the component in index 0.
 *
 * @param domainEndpoint The Elasticsearch domain endpoint
 */
function extractNameFromEndpoint(domainEndpoint) {
    const { hostname } = new url_1.URL(domainEndpoint);
    const domain = hostname.split('.')[0];
    const suffix = '-' + domain.split('-').slice(-1)[0];
    return domain.split(suffix)[0];
}
/**
 * Converts an Elasticsearch version into a into a decimal number with major and minor version i.e x.y.
 *
 * @param version The Elasticsearch version object
 */
function parseVersion(version) {
    const versionStr = version.version;
    const firstDot = versionStr.indexOf('.');
    if (firstDot < 1) {
        throw new Error(`Invalid Elasticsearch version: ${versionStr}. Version string needs to start with major and minor version (x.y).`);
    }
    const secondDot = versionStr.indexOf('.', firstDot + 1);
    try {
        if (secondDot == -1) {
            return parseFloat(versionStr);
        }
        else {
            return parseFloat(versionStr.substring(0, secondDot));
        }
    }
    catch (error) {
        throw new Error(`Invalid Elasticsearch version: ${versionStr}. Version string needs to start with major and minor version (x.y).`);
    }
}
//# sourceMappingURL=data:application/json;base64,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