import { Resource } from "../../core";
import { Construct } from 'constructs';
import { IEventBus } from './event-bus';
import { EventPattern } from './event-pattern';
import { IRule } from './rule-ref';
import { Schedule } from './schedule';
import { IRuleTarget } from './target';
/**
 * Properties for defining an EventBridge Rule
 */
export interface RuleProps {
    /**
     * A description of the rule's purpose.
     *
     * @default - No description.
     */
    readonly description?: string;
    /**
     * A name for the rule.
     *
     * @default - AWS CloudFormation generates a unique physical ID and uses that ID
     * for the rule name. For more information, see Name Type.
     */
    readonly ruleName?: string;
    /**
     * Indicates whether the rule is enabled.
     *
     * @default true
     */
    readonly enabled?: boolean;
    /**
     * The schedule or rate (frequency) that determines when EventBridge
     * runs the rule. For more information, see Schedule Expression Syntax for
     * Rules in the Amazon EventBridge User Guide.
     *
     * @see https://docs.aws.amazon.com/eventbridge/latest/userguide/scheduled-events.html
     *
     * You must specify this property, the `eventPattern` property, or both.
     *
     * @default - None.
     */
    readonly schedule?: Schedule;
    /**
     * Describes which events EventBridge routes to the specified target.
     * These routed events are matched events. For more information, see Events
     * and Event Patterns in the Amazon EventBridge User Guide.
     *
     * @see
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/eventbridge-and-event-patterns.html
     *
     * You must specify this property (either via props or via
     * `addEventPattern`), the `scheduleExpression` property, or both. The
     * method `addEventPattern` can be used to add filter values to the event
     * pattern.
     *
     * @default - None.
     */
    readonly eventPattern?: EventPattern;
    /**
     * Targets to invoke when this rule matches an event.
     *
     * Input will be the full matched event. If you wish to specify custom
     * target input, use `addTarget(target[, inputOptions])`.
     *
     * @default - No targets.
     */
    readonly targets?: IRuleTarget[];
    /**
     * The event bus to associate with this rule.
     *
     * @default - The default event bus.
     */
    readonly eventBus?: IEventBus;
}
/**
 * Defines an EventBridge Rule in this stack.
 *
 * @resource AWS::Events::Rule
 */
export declare class Rule extends Resource implements IRule {
    /**
     * Import an existing EventBridge Rule provided an ARN
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param eventRuleArn Event Rule ARN (i.e. arn:aws:events:<region>:<account-id>:rule/MyScheduledRule).
     */
    static fromEventRuleArn(scope: Construct, id: string, eventRuleArn: string): IRule;
    readonly ruleArn: string;
    readonly ruleName: string;
    private readonly targets;
    private readonly eventPattern;
    private readonly scheduleExpression?;
    private readonly description?;
    private readonly accountEventBusTargets;
    constructor(scope: Construct, id: string, props?: RuleProps);
    /**
     * Adds a target to the rule. The abstract class RuleTarget can be extended to define new
     * targets.
     *
     * No-op if target is undefined.
     */
    addTarget(target?: IRuleTarget): void;
    /**
     * Adds an event pattern filter to this rule. If a pattern was already specified,
     * these values are merged into the existing pattern.
     *
     * For example, if the rule already contains the pattern:
     *
     *    {
     *      "resources": [ "r1" ],
     *      "detail": {
     *        "hello": [ 1 ]
     *      }
     *    }
     *
     * And `addEventPattern` is called with the pattern:
     *
     *    {
     *      "resources": [ "r2" ],
     *      "detail": {
     *        "foo": [ "bar" ]
     *      }
     *    }
     *
     * The resulting event pattern will be:
     *
     *    {
     *      "resources": [ "r1", "r2" ],
     *      "detail": {
     *        "hello": [ 1 ],
     *        "foo": [ "bar" ]
     *      }
     *    }
     *
     */
    addEventPattern(eventPattern?: EventPattern): void;
    protected validate(): string[];
    private renderTargets;
}
