"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Rule = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const constructs_1 = require("constructs");
const events_generated_1 = require("./events.generated");
const schedule_1 = require("./schedule");
const util_1 = require("./util");
/**
 * Defines an EventBridge Rule in this stack.
 *
 * @resource AWS::Events::Rule
 */
class Rule extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.ruleName,
        });
        this.targets = new Array();
        this.eventPattern = {};
        this.accountEventBusTargets = {};
        if (props.eventBus && props.schedule) {
            throw new Error('Cannot associate rule with \'eventBus\' when using \'schedule\'');
        }
        this.description = props.description;
        this.scheduleExpression = props.schedule && props.schedule.expressionString;
        const resource = new events_generated_1.CfnRule(this, 'Resource', {
            name: this.physicalName,
            description: this.description,
            state: props.enabled == null ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
            scheduleExpression: this.scheduleExpression,
            eventPattern: core_1.Lazy.anyValue({ produce: () => this._renderEventPattern() }),
            targets: core_1.Lazy.anyValue({ produce: () => this.renderTargets() }),
            eventBusName: props.eventBus && props.eventBus.eventBusName,
        });
        this.ruleArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'events',
            resource: 'rule',
            resourceName: this.physicalName,
        });
        this.ruleName = this.getResourceNameAttribute(resource.ref);
        this.addEventPattern(props.eventPattern);
        for (const target of props.targets || []) {
            this.addTarget(target);
        }
    }
    /**
     * Import an existing EventBridge Rule provided an ARN
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param eventRuleArn Event Rule ARN (i.e. arn:aws:events:<region>:<account-id>:rule/MyScheduledRule).
     */
    static fromEventRuleArn(scope, id, eventRuleArn) {
        const parts = core_1.Stack.of(scope).parseArn(eventRuleArn);
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.ruleArn = eventRuleArn;
                this.ruleName = parts.resourceName || '';
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds a target to the rule. The abstract class RuleTarget can be extended to define new
     * targets.
     *
     * No-op if target is undefined.
     */
    addTarget(target) {
        if (!target) {
            return;
        }
        // Simply increment id for each `addTarget` call. This is guaranteed to be unique.
        const autoGeneratedId = `Target${this.targets.length}`;
        const targetProps = target.bind(this, autoGeneratedId);
        const inputProps = targetProps.input && targetProps.input.bind(this);
        const roleArn = targetProps.role ? targetProps.role.roleArn : undefined;
        const id = targetProps.id || autoGeneratedId;
        if (targetProps.targetResource) {
            const targetStack = core_1.Stack.of(targetProps.targetResource);
            const targetAccount = targetStack.account;
            const targetRegion = targetStack.region;
            const sourceStack = core_1.Stack.of(this);
            const sourceAccount = sourceStack.account;
            const sourceRegion = sourceStack.region;
            if (targetRegion !== sourceRegion) {
                throw new Error('Rule and target must be in the same region');
            }
            if (targetAccount !== sourceAccount) {
                // cross-account event - strap in, this works differently than regular events!
                // based on:
                // https://docs.aws.amazon.com/eventbridge/latest/userguide/eventbridge-cross-account-event-delivery.html
                // for cross-account events, we require concrete accounts
                if (core_1.Token.isUnresolved(targetAccount)) {
                    throw new Error('You need to provide a concrete account for the target stack when using cross-account events');
                }
                if (core_1.Token.isUnresolved(sourceAccount)) {
                    throw new Error('You need to provide a concrete account for the source stack when using cross-account events');
                }
                // and the target region has to be concrete as well
                if (core_1.Token.isUnresolved(targetRegion)) {
                    throw new Error('You need to provide a concrete region for the target stack when using cross-account events');
                }
                // the _actual_ target is just the event bus of the target's account
                // make sure we only add it once per account
                const exists = this.accountEventBusTargets[targetAccount];
                if (!exists) {
                    this.accountEventBusTargets[targetAccount] = true;
                    this.targets.push({
                        id,
                        arn: targetStack.formatArn({
                            service: 'events',
                            resource: 'event-bus',
                            resourceName: 'default',
                            region: targetRegion,
                            account: targetAccount,
                        }),
                    });
                }
                // Grant the source account permissions to publish events to the event bus of the target account.
                // Do it in a separate stack instead of the target stack (which seems like the obvious place to put it),
                // because it needs to be deployed before the rule containing the above event-bus target in the source stack
                // (EventBridge verifies whether you have permissions to the targets on rule creation),
                // but it's common for the target stack to depend on the source stack
                // (that's the case with CodePipeline, for example)
                const sourceApp = this.node.root;
                if (!sourceApp || !core_1.App.isApp(sourceApp)) {
                    throw new Error('Event stack which uses cross-account targets must be part of a CDK app');
                }
                const targetApp = constructs_1.Node.of(targetProps.targetResource).root;
                if (!targetApp || !core_1.App.isApp(targetApp)) {
                    throw new Error('Target stack which uses cross-account event targets must be part of a CDK app');
                }
                if (sourceApp !== targetApp) {
                    throw new Error('Event stack and target stack must belong to the same CDK app');
                }
                const stackId = `EventBusPolicy-${sourceAccount}-${targetRegion}-${targetAccount}`;
                let eventBusPolicyStack = sourceApp.node.tryFindChild(stackId);
                if (!eventBusPolicyStack) {
                    eventBusPolicyStack = new core_1.Stack(sourceApp, stackId, {
                        env: {
                            account: targetAccount,
                            region: targetRegion,
                        },
                        stackName: `${targetStack.stackName}-EventBusPolicy-support-${targetRegion}-${sourceAccount}`,
                    });
                    new events_generated_1.CfnEventBusPolicy(eventBusPolicyStack, 'GivePermToOtherAccount', {
                        action: 'events:PutEvents',
                        statementId: `Allow-account-${sourceAccount}`,
                        principal: sourceAccount,
                    });
                }
                // deploy the event bus permissions before the source stack
                sourceStack.addDependency(eventBusPolicyStack);
                // The actual rule lives in the target stack.
                // Other than the account, it's identical to this one
                // eventPattern is mutable through addEventPattern(), so we need to lazy evaluate it
                // but only Tokens can be lazy in the framework, so make a subclass instead
                const self = this;
                class CopyRule extends Rule {
                    _renderEventPattern() {
                        return self._renderEventPattern();
                    }
                    // we need to override validate(), as it uses the
                    // value of the eventPattern field,
                    // which might be empty in the case of the copied rule
                    // (as the patterns in the original might be added through addEventPattern(),
                    // not passed through the constructor).
                    // Anyway, even if the original rule is invalid,
                    // we would get duplicate errors if we didn't override this,
                    // which is probably a bad idea in and of itself
                    validate() {
                        return [];
                    }
                }
                new CopyRule(targetStack, `${this.node.uniqueId}-${id}`, {
                    targets: [target],
                    eventPattern: this.eventPattern,
                    schedule: this.scheduleExpression ? schedule_1.Schedule.expression(this.scheduleExpression) : undefined,
                    description: this.description,
                });
                return;
            }
        }
        this.targets.push({
            id,
            arn: targetProps.arn,
            roleArn,
            ecsParameters: targetProps.ecsParameters,
            kinesisParameters: targetProps.kinesisParameters,
            runCommandParameters: targetProps.runCommandParameters,
            batchParameters: targetProps.batchParameters,
            sqsParameters: targetProps.sqsParameters,
            input: inputProps && inputProps.input,
            inputPath: inputProps && inputProps.inputPath,
            inputTransformer: inputProps && inputProps.inputTemplate !== undefined ? {
                inputTemplate: inputProps.inputTemplate,
                inputPathsMap: inputProps.inputPathsMap,
            } : undefined,
        });
    }
    /**
     * Adds an event pattern filter to this rule. If a pattern was already specified,
     * these values are merged into the existing pattern.
     *
     * For example, if the rule already contains the pattern:
     *
     *    {
     *      "resources": [ "r1" ],
     *      "detail": {
     *        "hello": [ 1 ]
     *      }
     *    }
     *
     * And `addEventPattern` is called with the pattern:
     *
     *    {
     *      "resources": [ "r2" ],
     *      "detail": {
     *        "foo": [ "bar" ]
     *      }
     *    }
     *
     * The resulting event pattern will be:
     *
     *    {
     *      "resources": [ "r1", "r2" ],
     *      "detail": {
     *        "hello": [ 1 ],
     *        "foo": [ "bar" ]
     *      }
     *    }
     *
     */
    addEventPattern(eventPattern) {
        if (!eventPattern) {
            return;
        }
        util_1.mergeEventPattern(this.eventPattern, eventPattern);
    }
    /**
     * Not private only to be overrideen in CopyRule.
     *
     * @internal
     */
    _renderEventPattern() {
        const eventPattern = this.eventPattern;
        if (Object.keys(eventPattern).length === 0) {
            return undefined;
        }
        // rename 'detailType' to 'detail-type'
        const out = {};
        for (let key of Object.keys(eventPattern)) {
            const value = eventPattern[key];
            if (key === 'detailType') {
                key = 'detail-type';
            }
            out[key] = value;
        }
        return out;
    }
    validate() {
        if (Object.keys(this.eventPattern).length === 0 && !this.scheduleExpression) {
            return ['Either \'eventPattern\' or \'schedule\' must be defined'];
        }
        return [];
    }
    renderTargets() {
        if (this.targets.length === 0) {
            return undefined;
        }
        return this.targets;
    }
}
exports.Rule = Rule;
//# sourceMappingURL=data:application/json;base64,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