"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Schedule = void 0;
/**
 * Schedule for scheduled event rules
 */
class Schedule {
    constructor() {
    }
    /**
     * Construct a schedule from a literal schedule expression
     *
     * @param expression The expression to use. Must be in a format that EventBridge will recognize
     */
    static expression(expression) {
        return new LiteralSchedule(expression);
    }
    /**
     * Construct a schedule from an interval and a time unit
     */
    static rate(duration) {
        if (duration.toSeconds() === 0) {
            throw new Error('Duration cannot be 0');
        }
        let rate = maybeRate(duration.toDays({ integral: false }), 'day');
        if (rate === undefined) {
            rate = maybeRate(duration.toHours({ integral: false }), 'hour');
        }
        if (rate === undefined) {
            rate = makeRate(duration.toMinutes({ integral: true }), 'minute');
        }
        return new LiteralSchedule(rate);
    }
    /**
     * Create a schedule from a set of cron fields
     */
    static cron(options) {
        if (options.weekDay !== undefined && options.day !== undefined) {
            throw new Error('Cannot supply both \'day\' and \'weekDay\', use at most one');
        }
        const minute = fallback(options.minute, '*');
        const hour = fallback(options.hour, '*');
        const month = fallback(options.month, '*');
        const year = fallback(options.year, '*');
        // Weekday defaults to '?' if not supplied. If it is supplied, day must become '?'
        const day = fallback(options.day, options.weekDay !== undefined ? '?' : '*');
        const weekDay = fallback(options.weekDay, '?');
        return new LiteralSchedule(`cron(${minute} ${hour} ${day} ${month} ${weekDay} ${year})`);
    }
}
exports.Schedule = Schedule;
class LiteralSchedule extends Schedule {
    constructor(expressionString) {
        super();
        this.expressionString = expressionString;
    }
}
function fallback(x, def) {
    return x === undefined ? def : x;
}
/**
 * Return the rate if the rate is whole number
 */
function maybeRate(interval, singular) {
    if (interval === 0 || !Number.isInteger(interval)) {
        return undefined;
    }
    return makeRate(interval, singular);
}
/**
 * Return 'rate(${interval} ${singular}(s))` for the interval
 */
function makeRate(interval, singular) {
    return interval === 1 ? `rate(1 ${singular})` : `rate(${interval} ${singular}s)`;
}
//# sourceMappingURL=data:application/json;base64,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