"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Database = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const glue_generated_1 = require("./glue.generated");
/**
 * A Glue database.
 */
class Database extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.databaseName,
        });
        let databaseInput = {
            name: props.databaseName,
        };
        if (props.locationUri !== undefined) {
            validateLocationUri(props.locationUri);
            this.locationUri = props.locationUri;
            databaseInput = {
                locationUri: this.locationUri,
                ...databaseInput,
            };
        }
        this.catalogId = core_1.Stack.of(this).account;
        const resource = new glue_generated_1.CfnDatabase(this, 'Resource', {
            catalogId: this.catalogId,
            databaseInput,
        });
        // see https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html#data-catalog-resource-arns
        this.databaseName = this.getResourceNameAttribute(resource.ref);
        this.databaseArn = this.stack.formatArn({
            service: 'glue',
            resource: 'database',
            resourceName: this.databaseName,
        });
        // catalogId is implicitly the accountId, which is why we don't pass the catalogId here
        this.catalogArn = core_1.Stack.of(this).formatArn({
            service: 'glue',
            resource: 'catalog',
        });
    }
    static fromDatabaseArn(scope, id, databaseArn) {
        const stack = core_1.Stack.of(scope);
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.databaseArn = databaseArn;
                this.databaseName = stack.parseArn(databaseArn).resourceName;
                this.catalogArn = stack.formatArn({ service: 'glue', resource: 'catalog' });
                this.catalogId = stack.account;
            }
        }
        return new Import(scope, id);
    }
}
exports.Database = Database;
function validateLocationUri(locationUri) {
    if (locationUri.length < 1 || locationUri.length > 1024) {
        throw new Error(`locationUri length must be (inclusively) between 1 and 1024, but was ${locationUri.length}`);
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGF0YWJhc2UuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJkYXRhYmFzZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxxQ0FBd0QsQ0FBQyxnREFBZ0Q7QUFFekcscURBQStDO0FBb0MvQzs7R0FFRztBQUNILE1BQWEsUUFBUyxTQUFRLGVBQVE7SUErQmxDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBb0I7UUFDMUQsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDYixZQUFZLEVBQUUsS0FBSyxDQUFDLFlBQVk7U0FDbkMsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxhQUFhLEdBQXNDO1lBQ25ELElBQUksRUFBRSxLQUFLLENBQUMsWUFBWTtTQUMzQixDQUFDO1FBQ0YsSUFBSSxLQUFLLENBQUMsV0FBVyxLQUFLLFNBQVMsRUFBRTtZQUNqQyxtQkFBbUIsQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLENBQUM7WUFDdkMsSUFBSSxDQUFDLFdBQVcsR0FBRyxLQUFLLENBQUMsV0FBVyxDQUFDO1lBQ3JDLGFBQWEsR0FBRztnQkFDWixXQUFXLEVBQUUsSUFBSSxDQUFDLFdBQVc7Z0JBQzdCLEdBQUcsYUFBYTthQUNuQixDQUFDO1NBQ0w7UUFDRCxJQUFJLENBQUMsU0FBUyxHQUFHLFlBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDO1FBQ3hDLE1BQU0sUUFBUSxHQUFHLElBQUksNEJBQVcsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQy9DLFNBQVMsRUFBRSxJQUFJLENBQUMsU0FBUztZQUN6QixhQUFhO1NBQ2hCLENBQUMsQ0FBQztRQUNILCtHQUErRztRQUMvRyxJQUFJLENBQUMsWUFBWSxHQUFHLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDaEUsSUFBSSxDQUFDLFdBQVcsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQztZQUNwQyxPQUFPLEVBQUUsTUFBTTtZQUNmLFFBQVEsRUFBRSxVQUFVO1lBQ3BCLFlBQVksRUFBRSxJQUFJLENBQUMsWUFBWTtTQUNsQyxDQUFDLENBQUM7UUFDSCx1RkFBdUY7UUFDdkYsSUFBSSxDQUFDLFVBQVUsR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFNBQVMsQ0FBQztZQUN2QyxPQUFPLEVBQUUsTUFBTTtZQUNmLFFBQVEsRUFBRSxTQUFTO1NBQ3RCLENBQUMsQ0FBQztJQUNQLENBQUM7SUE5RE0sTUFBTSxDQUFDLGVBQWUsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxXQUFtQjtRQUMzRSxNQUFNLEtBQUssR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQzlCLE1BQU0sTUFBTyxTQUFRLGVBQVE7WUFBN0I7O2dCQUNXLGdCQUFXLEdBQUcsV0FBVyxDQUFDO2dCQUMxQixpQkFBWSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLENBQUMsWUFBYSxDQUFDO2dCQUN6RCxlQUFVLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQyxFQUFFLE9BQU8sRUFBRSxNQUFNLEVBQUUsUUFBUSxFQUFFLFNBQVMsRUFBRSxDQUFDLENBQUM7Z0JBQ3ZFLGNBQVMsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1lBQ3JDLENBQUM7U0FBQTtRQUNELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQ2pDLENBQUM7Q0FzREo7QUFoRUQsNEJBZ0VDO0FBQ0QsU0FBUyxtQkFBbUIsQ0FBQyxXQUFtQjtJQUM1QyxJQUFJLFdBQVcsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxJQUFJLFdBQVcsQ0FBQyxNQUFNLEdBQUcsSUFBSSxFQUFFO1FBQ3JELE1BQU0sSUFBSSxLQUFLLENBQUMsd0VBQXdFLFdBQVcsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDO0tBQ2pIO0FBQ0wsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IElSZXNvdXJjZSwgUmVzb3VyY2UsIFN0YWNrIH0gZnJvbSBcIi4uLy4uL2NvcmVcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2NvcmUnXG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IENmbkRhdGFiYXNlIH0gZnJvbSAnLi9nbHVlLmdlbmVyYXRlZCc7XG5leHBvcnQgaW50ZXJmYWNlIElEYXRhYmFzZSBleHRlbmRzIElSZXNvdXJjZSB7XG4gICAgLyoqXG4gICAgICogVGhlIEFSTiBvZiB0aGUgY2F0YWxvZy5cbiAgICAgKi9cbiAgICByZWFkb25seSBjYXRhbG9nQXJuOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIGNhdGFsb2cgaWQgb2YgdGhlIGRhdGFiYXNlICh1c3VhbGx5LCB0aGUgQVdTIGFjY291bnQgaWQpXG4gICAgICovXG4gICAgcmVhZG9ubHkgY2F0YWxvZ0lkOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIEFSTiBvZiB0aGUgZGF0YWJhc2UuXG4gICAgICpcbiAgICAgKiBAYXR0cmlidXRlXG4gICAgICovXG4gICAgcmVhZG9ubHkgZGF0YWJhc2VBcm46IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgbmFtZSBvZiB0aGUgZGF0YWJhc2UuXG4gICAgICpcbiAgICAgKiBAYXR0cmlidXRlXG4gICAgICovXG4gICAgcmVhZG9ubHkgZGF0YWJhc2VOYW1lOiBzdHJpbmc7XG59XG5leHBvcnQgaW50ZXJmYWNlIERhdGFiYXNlUHJvcHMge1xuICAgIC8qKlxuICAgICAqIFRoZSBuYW1lIG9mIHRoZSBkYXRhYmFzZS5cbiAgICAgKi9cbiAgICByZWFkb25seSBkYXRhYmFzZU5hbWU6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgbG9jYXRpb24gb2YgdGhlIGRhdGFiYXNlIChmb3IgZXhhbXBsZSwgYW4gSERGUyBwYXRoKS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IHVuZGVmaW5lZC4gVGhpcyBmaWVsZCBpcyBvcHRpb25hbCBpbiBBV1M6OkdsdWU6OkRhdGFiYXNlIERhdGFiYXNlSW5wdXRcbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NDbG91ZEZvcm1hdGlvbi9sYXRlc3QvVXNlckd1aWRlL2F3cy1wcm9wZXJ0aWVzLWdsdWUtZGF0YWJhc2UtZGF0YWJhc2VpbnB1dC5odG1sXG4gICAgICovXG4gICAgcmVhZG9ubHkgbG9jYXRpb25Vcmk/OiBzdHJpbmc7XG59XG4vKipcbiAqIEEgR2x1ZSBkYXRhYmFzZS5cbiAqL1xuZXhwb3J0IGNsYXNzIERhdGFiYXNlIGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJRGF0YWJhc2Uge1xuICAgIHB1YmxpYyBzdGF0aWMgZnJvbURhdGFiYXNlQXJuKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGRhdGFiYXNlQXJuOiBzdHJpbmcpOiBJRGF0YWJhc2Uge1xuICAgICAgICBjb25zdCBzdGFjayA9IFN0YWNrLm9mKHNjb3BlKTtcbiAgICAgICAgY2xhc3MgSW1wb3J0IGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJRGF0YWJhc2Uge1xuICAgICAgICAgICAgcHVibGljIGRhdGFiYXNlQXJuID0gZGF0YWJhc2VBcm47XG4gICAgICAgICAgICBwdWJsaWMgZGF0YWJhc2VOYW1lID0gc3RhY2sucGFyc2VBcm4oZGF0YWJhc2VBcm4pLnJlc291cmNlTmFtZSE7XG4gICAgICAgICAgICBwdWJsaWMgY2F0YWxvZ0FybiA9IHN0YWNrLmZvcm1hdEFybih7IHNlcnZpY2U6ICdnbHVlJywgcmVzb3VyY2U6ICdjYXRhbG9nJyB9KTtcbiAgICAgICAgICAgIHB1YmxpYyBjYXRhbG9nSWQgPSBzdGFjay5hY2NvdW50O1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBuZXcgSW1wb3J0KHNjb3BlLCBpZCk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFSTiBvZiB0aGUgR2x1ZSBjYXRhbG9nIGluIHdoaWNoIHRoaXMgZGF0YWJhc2UgaXMgc3RvcmVkLlxuICAgICAqL1xuICAgIHB1YmxpYyByZWFkb25seSBjYXRhbG9nQXJuOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogSUQgb2YgdGhlIEdsdWUgY2F0YWxvZyBpbiB3aGljaCB0aGlzIGRhdGFiYXNlIGlzIHN0b3JlZC5cbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgY2F0YWxvZ0lkOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogQVJOIG9mIHRoaXMgZGF0YWJhc2UuXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IGRhdGFiYXNlQXJuOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogTmFtZSBvZiB0aGlzIGRhdGFiYXNlLlxuICAgICAqL1xuICAgIHB1YmxpYyByZWFkb25seSBkYXRhYmFzZU5hbWU6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBMb2NhdGlvbiBVUkkgb2YgdGhpcyBkYXRhYmFzZS5cbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgbG9jYXRpb25Vcmk/OiBzdHJpbmc7XG4gICAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IERhdGFiYXNlUHJvcHMpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkLCB7XG4gICAgICAgICAgICBwaHlzaWNhbE5hbWU6IHByb3BzLmRhdGFiYXNlTmFtZSxcbiAgICAgICAgfSk7XG4gICAgICAgIGxldCBkYXRhYmFzZUlucHV0OiBDZm5EYXRhYmFzZS5EYXRhYmFzZUlucHV0UHJvcGVydHkgPSB7XG4gICAgICAgICAgICBuYW1lOiBwcm9wcy5kYXRhYmFzZU5hbWUsXG4gICAgICAgIH07XG4gICAgICAgIGlmIChwcm9wcy5sb2NhdGlvblVyaSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgICB2YWxpZGF0ZUxvY2F0aW9uVXJpKHByb3BzLmxvY2F0aW9uVXJpKTtcbiAgICAgICAgICAgIHRoaXMubG9jYXRpb25VcmkgPSBwcm9wcy5sb2NhdGlvblVyaTtcbiAgICAgICAgICAgIGRhdGFiYXNlSW5wdXQgPSB7XG4gICAgICAgICAgICAgICAgbG9jYXRpb25Vcmk6IHRoaXMubG9jYXRpb25VcmksXG4gICAgICAgICAgICAgICAgLi4uZGF0YWJhc2VJbnB1dCxcbiAgICAgICAgICAgIH07XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy5jYXRhbG9nSWQgPSBTdGFjay5vZih0aGlzKS5hY2NvdW50O1xuICAgICAgICBjb25zdCByZXNvdXJjZSA9IG5ldyBDZm5EYXRhYmFzZSh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICAgICAgICBjYXRhbG9nSWQ6IHRoaXMuY2F0YWxvZ0lkLFxuICAgICAgICAgICAgZGF0YWJhc2VJbnB1dCxcbiAgICAgICAgfSk7XG4gICAgICAgIC8vIHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ2x1ZS9sYXRlc3QvZGcvZ2x1ZS1zcGVjaWZ5aW5nLXJlc291cmNlLWFybnMuaHRtbCNkYXRhLWNhdGFsb2ctcmVzb3VyY2UtYXJuc1xuICAgICAgICB0aGlzLmRhdGFiYXNlTmFtZSA9IHRoaXMuZ2V0UmVzb3VyY2VOYW1lQXR0cmlidXRlKHJlc291cmNlLnJlZik7XG4gICAgICAgIHRoaXMuZGF0YWJhc2VBcm4gPSB0aGlzLnN0YWNrLmZvcm1hdEFybih7XG4gICAgICAgICAgICBzZXJ2aWNlOiAnZ2x1ZScsXG4gICAgICAgICAgICByZXNvdXJjZTogJ2RhdGFiYXNlJyxcbiAgICAgICAgICAgIHJlc291cmNlTmFtZTogdGhpcy5kYXRhYmFzZU5hbWUsXG4gICAgICAgIH0pO1xuICAgICAgICAvLyBjYXRhbG9nSWQgaXMgaW1wbGljaXRseSB0aGUgYWNjb3VudElkLCB3aGljaCBpcyB3aHkgd2UgZG9uJ3QgcGFzcyB0aGUgY2F0YWxvZ0lkIGhlcmVcbiAgICAgICAgdGhpcy5jYXRhbG9nQXJuID0gU3RhY2sub2YodGhpcykuZm9ybWF0QXJuKHtcbiAgICAgICAgICAgIHNlcnZpY2U6ICdnbHVlJyxcbiAgICAgICAgICAgIHJlc291cmNlOiAnY2F0YWxvZycsXG4gICAgICAgIH0pO1xuICAgIH1cbn1cbmZ1bmN0aW9uIHZhbGlkYXRlTG9jYXRpb25VcmkobG9jYXRpb25Vcmk6IHN0cmluZyk6IHZvaWQge1xuICAgIGlmIChsb2NhdGlvblVyaS5sZW5ndGggPCAxIHx8IGxvY2F0aW9uVXJpLmxlbmd0aCA+IDEwMjQpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBsb2NhdGlvblVyaSBsZW5ndGggbXVzdCBiZSAoaW5jbHVzaXZlbHkpIGJldHdlZW4gMSBhbmQgMTAyNCwgYnV0IHdhcyAke2xvY2F0aW9uVXJpLmxlbmd0aH1gKTtcbiAgICB9XG59XG4iXX0=