"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Schema = void 0;
/**
 * @see https://docs.aws.amazon.com/athena/latest/ug/data-types.html
 */
class Schema {
    /**
     * Creates a decimal type.
     *
     * TODO: Bounds
     *
     * @param precision the total number of digits
     * @param scale the number of digits in fractional part, the default is 0
     */
    static decimal(precision, scale) {
        return {
            isPrimitive: true,
            inputString: scale !== undefined ? `decimal(${precision},${scale})` : `decimal(${precision})`,
        };
    }
    /**
     * Fixed length character data, with a specified length between 1 and 255.
     *
     * @param length length between 1 and 255
     */
    static char(length) {
        if (length <= 0 || length > 255) {
            throw new Error(`char length must be (inclusively) between 1 and 255, but was ${length}`);
        }
        if (length % 1 !== 0) {
            throw new Error(`char length must be a positive integer, was ${length}`);
        }
        return {
            isPrimitive: true,
            inputString: `char(${length})`,
        };
    }
    /**
     * Variable length character data, with a specified length between 1 and 65535.
     *
     * @param length length between 1 and 65535.
     */
    static varchar(length) {
        if (length <= 0 || length > 65535) {
            throw new Error(`varchar length must be (inclusively) between 1 and 65535, but was ${length}`);
        }
        if (length % 1 !== 0) {
            throw new Error(`varchar length must be a positive integer, was ${length}`);
        }
        return {
            isPrimitive: true,
            inputString: `varchar(${length})`,
        };
    }
    /**
     * Creates an array of some other type.
     *
     * @param itemType type contained by the array.
     */
    static array(itemType) {
        return {
            isPrimitive: false,
            inputString: `array<${itemType.inputString}>`,
        };
    }
    /**
     * Creates a map of some primitive key type to some value type.
     *
     * @param keyType type of key, must be a primitive.
     * @param valueType type fo the value indexed by the key.
     */
    static map(keyType, valueType) {
        if (!keyType.isPrimitive) {
            throw new Error(`the key type of a 'map' must be a primitive, but was ${keyType.inputString}`);
        }
        return {
            isPrimitive: false,
            inputString: `map<${keyType.inputString},${valueType.inputString}>`,
        };
    }
    /**
     * Creates a nested structure containing individually named and typed columns.
     *
     * @param columns the columns of the structure.
     */
    static struct(columns) {
        return {
            isPrimitive: false,
            inputString: `struct<${columns.map(column => {
                if (column.comment === undefined) {
                    return `${column.name}:${column.type.inputString}`;
                }
                else {
                    return `${column.name}:${column.type.inputString} COMMENT '${column.comment}'`;
                }
            }).join(',')}>`,
        };
    }
}
exports.Schema = Schema;
Schema.BOOLEAN = {
    isPrimitive: true,
    inputString: 'boolean',
};
Schema.BINARY = {
    isPrimitive: true,
    inputString: 'binary',
};
/**
 * A 64-bit signed INTEGER in two’s complement format, with a minimum value of -2^63 and a maximum value of 2^63-1.
 */
Schema.BIG_INT = {
    isPrimitive: true,
    inputString: 'bigint',
};
Schema.DOUBLE = {
    isPrimitive: true,
    inputString: 'double',
};
Schema.FLOAT = {
    isPrimitive: true,
    inputString: 'float',
};
/**
 * A 32-bit signed INTEGER in two’s complement format, with a minimum value of -2^31 and a maximum value of 2^31-1.
 */
Schema.INTEGER = {
    isPrimitive: true,
    inputString: 'int',
};
/**
 * A 16-bit signed INTEGER in two’s complement format, with a minimum value of -2^15 and a maximum value of 2^15-1.
 */
Schema.SMALL_INT = {
    isPrimitive: true,
    inputString: 'smallint',
};
/**
 * A 8-bit signed INTEGER in two’s complement format, with a minimum value of -2^7 and a maximum value of 2^7-1
 */
Schema.TINY_INT = {
    isPrimitive: true,
    inputString: 'tinyint',
};
/**
 * Date type.
 */
Schema.DATE = {
    isPrimitive: true,
    inputString: 'date',
};
/**
 * Timestamp type (date and time).
 */
Schema.TIMESTAMP = {
    isPrimitive: true,
    inputString: 'timestamp',
};
/**
 * Arbitrary-length string type.
 */
Schema.STRING = {
    isPrimitive: true,
    inputString: 'string',
};
//# sourceMappingURL=data:application/json;base64,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