"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Policy = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const iam_generated_1 = require("./iam.generated");
const policy_document_1 = require("./policy-document");
const util_1 = require("./util");
/**
 * The AWS::IAM::Policy resource associates an IAM policy with IAM users, roles,
 * or groups. For more information about IAM policies, see [Overview of IAM
 * Policies](http://docs.aws.amazon.com/IAM/latest/UserGuide/policies_overview.html)
 * in the IAM User Guide guide.
 */
class Policy extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.policyName ||
                // generatePolicyName will take the last 128 characters of the logical id since
                // policy names are limited to 128. the last 8 chars are a stack-unique hash, so
                // that shouod be sufficient to ensure uniqueness within a principal.
                core_1.Lazy.stringValue({ produce: () => util_1.generatePolicyName(scope, resource.logicalId) }),
        });
        /**
         * The policy document.
         */
        this.document = new policy_document_1.PolicyDocument();
        this.roles = new Array();
        this.users = new Array();
        this.groups = new Array();
        this.referenceTaken = false;
        const self = this;
        class CfnPolicyConditional extends iam_generated_1.CfnPolicy {
            /**
             * This function returns `true` if the CFN resource should be included in
             * the cloudformation template unless `force` is `true`, if the policy
             * document is empty, the resource will not be included.
             */
            shouldSynthesize() {
                return self.force || self.referenceTaken || (!self.document.isEmpty && self.isAttached);
            }
        }
        const resource = new CfnPolicyConditional(this, 'Resource', {
            policyDocument: this.document,
            policyName: this.physicalName,
            roles: util_1.undefinedIfEmpty(() => this.roles.map(r => r.roleName)),
            users: util_1.undefinedIfEmpty(() => this.users.map(u => u.userName)),
            groups: util_1.undefinedIfEmpty(() => this.groups.map(g => g.groupName)),
        });
        this._policyName = this.physicalName;
        this.force = props.force !== undefined ? props.force : false;
        if (props.users) {
            props.users.forEach(u => this.attachToUser(u));
        }
        if (props.groups) {
            props.groups.forEach(g => this.attachToGroup(g));
        }
        if (props.roles) {
            props.roles.forEach(r => this.attachToRole(r));
        }
        if (props.statements) {
            props.statements.forEach(p => this.addStatements(p));
        }
    }
    /**
     * Import a policy in this app based on its name
     */
    static fromPolicyName(scope, id, policyName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.policyName = policyName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds a statement to the policy document.
     */
    addStatements(...statement) {
        this.document.addStatements(...statement);
    }
    /**
     * Attaches this policy to a user.
     */
    attachToUser(user) {
        if (this.users.find(u => u === user)) {
            return;
        }
        this.users.push(user);
        user.attachInlinePolicy(this);
    }
    /**
     * Attaches this policy to a role.
     */
    attachToRole(role) {
        if (this.roles.find(r => r === role)) {
            return;
        }
        this.roles.push(role);
        role.attachInlinePolicy(this);
    }
    /**
     * Attaches this policy to a group.
     */
    attachToGroup(group) {
        if (this.groups.find(g => g === group)) {
            return;
        }
        this.groups.push(group);
        group.attachInlinePolicy(this);
    }
    /**
     * The name of this policy.
     *
     * @attribute
     */
    get policyName() {
        this.referenceTaken = true;
        return this._policyName;
    }
    validate() {
        const result = new Array();
        // validate that the policy document is not empty
        if (this.document.isEmpty) {
            if (this.force) {
                result.push('Policy created with force=true is empty. You must add statements to the policy');
            }
            if (!this.force && this.referenceTaken) {
                result.push('This Policy has been referenced by a resource, so it must contain at least one statement.');
            }
        }
        // validate that the policy is attached to at least one principal (role, user or group).
        if (!this.isAttached) {
            if (this.force) {
                result.push('Policy created with force=true must be attached to at least one principal: user, group or role');
            }
            if (!this.force && this.referenceTaken) {
                result.push('This Policy has been referenced by a resource, so it must be attached to at least one user, group or role.');
            }
        }
        result.push(...this.document.validateForIdentityPolicy());
        return result;
    }
    /**
     * Whether the policy resource has been attached to any identity
     */
    get isAttached() {
        return this.groups.length + this.users.length + this.roles.length > 0;
    }
}
exports.Policy = Policy;
//# sourceMappingURL=data:application/json;base64,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