"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UnknownPrincipal = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const constructs_1 = require("constructs");
/**
 * A principal for use in resources that need to have a role but it's unknown
 *
 * Some resources have roles associated with them which they assume, such as
 * Lambda Functions, CodeBuild projects, StepFunctions machines, etc.
 *
 * When those resources are imported, their actual roles are not always
 * imported with them. When that happens, we use an instance of this class
 * instead, which will add user warnings when statements are attempted to be
 * added to it.
 */
class UnknownPrincipal {
    constructor(props) {
        this.assumeRoleAction = 'sts:AssumeRole';
        this.resource = props.resource;
        this.grantPrincipal = this;
    }
    get policyFragment() {
        throw new Error(`Cannot get policy fragment of ${constructs_1.Node.of(this.resource).path}, resource imported without a role`);
    }
    addToPrincipalPolicy(statement) {
        const stack = core_1.Stack.of(this.resource);
        const repr = JSON.stringify(stack.resolve(statement));
        core_1.Annotations.of(this.resource).addWarning(`Add statement to this resource's role: ${repr}`);
        // Pretend we did the work. The human will do it for us, eventually.
        return { statementAdded: true, policyDependable: new core_1.ConcreteDependable() };
    }
    addToPolicy(statement) {
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
}
exports.UnknownPrincipal = UnknownPrincipal;
//# sourceMappingURL=data:application/json;base64,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