"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.mergePrincipal = exports.AttachedPolicies = exports.generatePolicyName = exports.undefinedIfEmpty = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const MAX_POLICY_NAME_LEN = 128;
function undefinedIfEmpty(f) {
    return core_1.Lazy.listValue({
        produce: () => {
            const array = f();
            return (array && array.length > 0) ? array : undefined;
        },
    });
}
exports.undefinedIfEmpty = undefinedIfEmpty;
/**
 * Used to generate a unique policy name based on the policy resource construct.
 * The logical ID of the resource is a great candidate as long as it doesn't exceed
 * 128 characters, so we take the last 128 characters (in order to make sure the hash
 * is there).
 */
function generatePolicyName(scope, logicalId) {
    // as logicalId is itself a Token, resolve it first
    const resolvedLogicalId = core_1.Tokenization.resolve(logicalId, {
        scope,
        resolver: new core_1.DefaultTokenResolver(new core_1.StringConcat()),
    });
    return lastNCharacters(resolvedLogicalId, MAX_POLICY_NAME_LEN);
}
exports.generatePolicyName = generatePolicyName;
/**
 * Returns a string composed of the last n characters of str.
 * If str is shorter than n, returns str.
 *
 * @param str the string to return the last n characters of
 * @param n how many characters to return
 */
function lastNCharacters(str, n) {
    const startIndex = Math.max(str.length - n, 0);
    return str.substring(startIndex, str.length);
}
/**
 * Helper class that maintains the set of attached policies for a principal.
 */
class AttachedPolicies {
    constructor() {
        this.policies = new Array();
    }
    /**
     * Adds a policy to the list of attached policies.
     *
     * If this policy is already, attached, returns false.
     * If there is another policy attached with the same name, throws an exception.
     */
    attach(policy) {
        if (this.policies.find(p => p === policy)) {
            return; // already attached
        }
        if (this.policies.find(p => p.policyName === policy.policyName)) {
            throw new Error(`A policy named "${policy.policyName}" is already attached`);
        }
        this.policies.push(policy);
    }
}
exports.AttachedPolicies = AttachedPolicies;
/**
 * Merge two dictionaries that represent IAM principals
 */
function mergePrincipal(target, source) {
    for (const key of Object.keys(source)) {
        target[key] = target[key] || [];
        let value = source[key];
        if (!Array.isArray(value)) {
            value = [value];
        }
        target[key].push(...value);
    }
    return target;
}
exports.mergePrincipal = mergePrincipal;
//# sourceMappingURL=data:application/json;base64,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