import * as lambda from "../../aws-lambda";
import * as cdk from "../../core";
/**
 * Base options for Parcel bundling
 */
export interface ParcelBaseOptions {
    /**
     * Whether to minify files when bundling.
     *
     * @default false
     */
    readonly minify?: boolean;
    /**
     * Whether to include source maps when bundling.
     *
     * @default false
     */
    readonly sourceMaps?: boolean;
    /**
     * The cache directory (relative to the project root)
     *
     * Parcel uses a filesystem cache for fast rebuilds.
     *
     * @default - `.parcel-cache` in the working directory
     */
    readonly cacheDir?: string;
    /**
     * The root of the project. This will be used as the source for the volume
     * mounted in the Docker container. If you specify this prop, ensure that
     * this path includes `entry` and any module/dependencies used by your
     * function otherwise bundling will not be possible.
     *
     * @default - the closest path containing a .git folder
     */
    readonly projectRoot?: string;
    /**
     * Environment variables defined when Parcel runs.
     *
     * @default - no environment variables are defined.
     */
    readonly parcelEnvironment?: {
        [key: string]: string;
    };
    /**
     * A list of modules that should be considered as externals (already available
     * in the runtime).
     *
     * @default ['aws-sdk']
     */
    readonly externalModules?: string[];
    /**
     * A list of modules that should be installed instead of bundled. Modules are
     * installed in a Lambda compatible environnment.
     *
     * @default - all modules are bundled
     */
    readonly nodeModules?: string[];
    /**
     * The version of Parcel to use when running in a Docker container.
     *
     * @default - 2.0.0-beta.1
     */
    readonly parcelVersion?: string;
    /**
     * Build arguments to pass when building the bundling image.
     *
     * @default - no build arguments are passed
     */
    readonly buildArgs?: {
        [key: string]: string;
    };
    /**
     * Force bundling in a Docker container even if local bundling is
     * possible.This  is useful if your function relies on node modules
     * that should be installed (`nodeModules`) in a Lambda compatible
     * environment.
     *
     * @default false
     */
    readonly forceDockerBundling?: boolean;
    /**
     * A custom bundling Docker image.
     *
     * This image should have Parcel installed at `/`. If you plan to use `nodeModules`
     * it should also have `npm` or `yarn` depending on the lock file you're using.
     *
     * See https://github.com/aws/aws-cdk/blob/master/packages/%40aws-cdk/aws-lambda-nodejs/parcel/Dockerfile
     * for the default image provided by @aws-cdk/aws-lambda-nodejs.
     *
     * @default - use the Docker image provided by @aws-cdk/aws-lambda-nodejs
     */
    readonly bundlingDockerImage?: cdk.BundlingDockerImage;
}
/**
 * Options for Parcel bundling
 */
export interface ParcelOptions extends ParcelBaseOptions {
    /**
     * Entry file
     */
    readonly entry: string;
    /**
     * The runtime of the lambda function
     */
    readonly runtime: lambda.Runtime;
}
/**
 * Bundling
 */
export declare class Bundling {
    /**
     * Parcel bundled Lambda asset code
     */
    static parcel(options: ParcelOptions): lambda.AssetCode;
}
