"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const fs = require("fs");
const path = require("path");
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const bundlers_1 = require("./bundlers");
const package_json_manager_1 = require("./package-json-manager");
const util_1 = require("./util");
/**
 * Bundling
 */
class Bundling {
    /**
     * Parcel bundled Lambda asset code
     */
    static parcel(options) {
        var _a, _b, _c, _d, _e, _f, _g, _h;
        // Find project root
        const projectRoot = (_d = (_c = (_b = (_a = options.projectRoot) !== null && _a !== void 0 ? _a : util_1.findUp(`.git${path.sep}`)) !== null && _b !== void 0 ? _b : util_1.findUp(bundlers_1.LockFile.YARN)) !== null && _c !== void 0 ? _c : util_1.findUp(bundlers_1.LockFile.NPM)) !== null && _d !== void 0 ? _d : util_1.findUp('package.json');
        if (!projectRoot) {
            throw new Error('Cannot find project root. Please specify it with `projectRoot`.');
        }
        const relativeEntryPath = path.relative(projectRoot, path.resolve(options.entry));
        const packageJsonManager = new package_json_manager_1.PackageJsonManager(path.dirname(options.entry));
        // Collect external and install modules
        let includeNodeModules;
        let dependencies;
        const externalModules = (_e = options.externalModules) !== null && _e !== void 0 ? _e : ['aws-sdk'];
        if (externalModules || options.nodeModules) {
            const modules = [...externalModules, ...(_f = options.nodeModules) !== null && _f !== void 0 ? _f : []];
            includeNodeModules = {};
            for (const mod of modules) {
                includeNodeModules[mod] = false;
            }
            if (options.nodeModules) {
                dependencies = packageJsonManager.getVersions(options.nodeModules);
            }
        }
        let installer = bundlers_1.Installer.NPM;
        let lockFile;
        if (dependencies) {
            // Use npm unless we have a yarn.lock.
            if (fs.existsSync(path.join(projectRoot, bundlers_1.LockFile.YARN))) {
                installer = bundlers_1.Installer.YARN;
                lockFile = bundlers_1.LockFile.YARN;
            }
            else if (fs.existsSync(path.join(projectRoot, bundlers_1.LockFile.NPM))) {
                lockFile = bundlers_1.LockFile.NPM;
            }
        }
        // Configure target in package.json for Parcel
        packageJsonManager.update({
            targets: {
                'cdk-lambda': {
                    context: 'node',
                    includeNodeModules: includeNodeModules !== null && includeNodeModules !== void 0 ? includeNodeModules : true,
                    sourceMap: (_g = options.sourceMaps) !== null && _g !== void 0 ? _g : false,
                    minify: (_h = options.minify) !== null && _h !== void 0 ? _h : false,
                    engines: {
                        node: `>= ${runtimeVersion(options.runtime)}`,
                    },
                },
            },
        });
        // Local
        let localBundler;
        if (!options.forceDockerBundling) {
            localBundler = new bundlers_1.LocalBundler({
                projectRoot,
                relativeEntryPath,
                cacheDir: options.cacheDir,
                environment: options.parcelEnvironment,
                dependencies,
                installer,
                lockFile,
            });
        }
        // Docker
        const dockerBundler = new bundlers_1.DockerBundler({
            runtime: options.runtime,
            relativeEntryPath,
            cacheDir: options.cacheDir,
            environment: options.parcelEnvironment,
            bundlingDockerImage: options.bundlingDockerImage,
            buildImage: !bundlers_1.LocalBundler.runsLocally(projectRoot) || options.forceDockerBundling,
            buildArgs: options.buildArgs,
            parcelVersion: options.parcelVersion,
            dependencies,
            installer,
            lockFile,
        });
        return lambda.Code.fromAsset(projectRoot, {
            assetHashType: cdk.AssetHashType.OUTPUT,
            bundling: {
                local: localBundler,
                ...dockerBundler.bundlingOptions,
            },
        });
    }
}
exports.Bundling = Bundling;
function runtimeVersion(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return match[1];
}
//# sourceMappingURL=data:application/json;base64,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