"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PackageJsonManager = void 0;
const fs = require("fs");
const path = require("path");
const util_1 = require("./util");
/**
 * A package.json manager to act on the closest package.json file.
 *
 * Configuring the bundler requires to manipulate the package.json and then
 * restore it.
 */
class PackageJsonManager {
    constructor(directory) {
        const pkgPath = util_1.findUp('package.json', directory);
        if (!pkgPath) {
            throw new Error('Cannot find a `package.json` in this project.');
        }
        this.pkgPath = path.join(pkgPath, 'package.json');
        this.pkg = fs.readFileSync(this.pkgPath);
        this.pkgJson = JSON.parse(this.pkg.toString());
    }
    /**
     * Update the package.json
     */
    update(data) {
        fs.writeFileSync(this.pkgPath, JSON.stringify({
            ...this.pkgJson,
            ...data,
        }, null, 2));
    }
    /**
     * Restore the package.json to the original
     */
    restore() {
        fs.writeFileSync(this.pkgPath, this.pkg);
    }
    /**
     * Extract versions for a list of modules
     */
    getVersions(modules) {
        var _a, _b, _c;
        const dependencies = {};
        const allDependencies = {
            ...(_a = this.pkgJson.dependencies) !== null && _a !== void 0 ? _a : {},
            ...(_b = this.pkgJson.devDependencies) !== null && _b !== void 0 ? _b : {},
            ...(_c = this.pkgJson.peerDependencies) !== null && _c !== void 0 ? _c : {},
        };
        for (const mod of modules) {
            if (!allDependencies[mod]) {
                throw new Error(`Cannot extract version for ${mod} in package.json`);
            }
            dependencies[mod] = allDependencies[mod];
        }
        return dependencies;
    }
}
exports.PackageJsonManager = PackageJsonManager;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicGFja2FnZS1qc29uLW1hbmFnZXIuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJwYWNrYWdlLWpzb24tbWFuYWdlci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSx5QkFBeUI7QUFDekIsNkJBQTZCO0FBQzdCLGlDQUFnQztBQUNoQzs7Ozs7R0FLRztBQUNILE1BQWEsa0JBQWtCO0lBSTNCLFlBQVksU0FBaUI7UUFDekIsTUFBTSxPQUFPLEdBQUcsYUFBTSxDQUFDLGNBQWMsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUNsRCxJQUFJLENBQUMsT0FBTyxFQUFFO1lBQ1YsTUFBTSxJQUFJLEtBQUssQ0FBQywrQ0FBK0MsQ0FBQyxDQUFDO1NBQ3BFO1FBQ0QsSUFBSSxDQUFDLE9BQU8sR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxjQUFjLENBQUMsQ0FBQztRQUNsRCxJQUFJLENBQUMsR0FBRyxHQUFHLEVBQUUsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQ3pDLElBQUksQ0FBQyxPQUFPLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUM7SUFDbkQsQ0FBQztJQUNEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLElBQVM7UUFDbkIsRUFBRSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUM7WUFDMUMsR0FBRyxJQUFJLENBQUMsT0FBTztZQUNmLEdBQUcsSUFBSTtTQUNWLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDakIsQ0FBQztJQUNEOztPQUVHO0lBQ0ksT0FBTztRQUNWLEVBQUUsQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUNEOztPQUVHO0lBQ0ksV0FBVyxDQUFDLE9BQWlCOztRQUdoQyxNQUFNLFlBQVksR0FFZCxFQUFFLENBQUM7UUFDUCxNQUFNLGVBQWUsR0FBRztZQUNwQixTQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsWUFBWSxtQ0FBSSxFQUFFO1lBQ2xDLFNBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxlQUFlLG1DQUFJLEVBQUU7WUFDckMsU0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLGdCQUFnQixtQ0FBSSxFQUFFO1NBQ3pDLENBQUM7UUFDRixLQUFLLE1BQU0sR0FBRyxJQUFJLE9BQU8sRUFBRTtZQUN2QixJQUFJLENBQUMsZUFBZSxDQUFDLEdBQUcsQ0FBQyxFQUFFO2dCQUN2QixNQUFNLElBQUksS0FBSyxDQUFDLDhCQUE4QixHQUFHLGtCQUFrQixDQUFDLENBQUM7YUFDeEU7WUFDRCxZQUFZLENBQUMsR0FBRyxDQUFDLEdBQUcsZUFBZSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1NBQzVDO1FBQ0QsT0FBTyxZQUFZLENBQUM7SUFDeEIsQ0FBQztDQUNKO0FBbERELGdEQWtEQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGZzIGZyb20gJ2ZzJztcbmltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgeyBmaW5kVXAgfSBmcm9tICcuL3V0aWwnO1xuLyoqXG4gKiBBIHBhY2thZ2UuanNvbiBtYW5hZ2VyIHRvIGFjdCBvbiB0aGUgY2xvc2VzdCBwYWNrYWdlLmpzb24gZmlsZS5cbiAqXG4gKiBDb25maWd1cmluZyB0aGUgYnVuZGxlciByZXF1aXJlcyB0byBtYW5pcHVsYXRlIHRoZSBwYWNrYWdlLmpzb24gYW5kIHRoZW5cbiAqIHJlc3RvcmUgaXQuXG4gKi9cbmV4cG9ydCBjbGFzcyBQYWNrYWdlSnNvbk1hbmFnZXIge1xuICAgIHByaXZhdGUgcmVhZG9ubHkgcGtnUGF0aDogc3RyaW5nO1xuICAgIHByaXZhdGUgcmVhZG9ubHkgcGtnOiBCdWZmZXI7XG4gICAgcHJpdmF0ZSByZWFkb25seSBwa2dKc29uOiBQYWNrYWdlSnNvbjtcbiAgICBjb25zdHJ1Y3RvcihkaXJlY3Rvcnk6IHN0cmluZykge1xuICAgICAgICBjb25zdCBwa2dQYXRoID0gZmluZFVwKCdwYWNrYWdlLmpzb24nLCBkaXJlY3RvcnkpO1xuICAgICAgICBpZiAoIXBrZ1BhdGgpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IGZpbmQgYSBgcGFja2FnZS5qc29uYCBpbiB0aGlzIHByb2plY3QuJyk7XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy5wa2dQYXRoID0gcGF0aC5qb2luKHBrZ1BhdGgsICdwYWNrYWdlLmpzb24nKTtcbiAgICAgICAgdGhpcy5wa2cgPSBmcy5yZWFkRmlsZVN5bmModGhpcy5wa2dQYXRoKTtcbiAgICAgICAgdGhpcy5wa2dKc29uID0gSlNPTi5wYXJzZSh0aGlzLnBrZy50b1N0cmluZygpKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogVXBkYXRlIHRoZSBwYWNrYWdlLmpzb25cbiAgICAgKi9cbiAgICBwdWJsaWMgdXBkYXRlKGRhdGE6IGFueSkge1xuICAgICAgICBmcy53cml0ZUZpbGVTeW5jKHRoaXMucGtnUGF0aCwgSlNPTi5zdHJpbmdpZnkoe1xuICAgICAgICAgICAgLi4udGhpcy5wa2dKc29uLFxuICAgICAgICAgICAgLi4uZGF0YSxcbiAgICAgICAgfSwgbnVsbCwgMikpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBSZXN0b3JlIHRoZSBwYWNrYWdlLmpzb24gdG8gdGhlIG9yaWdpbmFsXG4gICAgICovXG4gICAgcHVibGljIHJlc3RvcmUoKSB7XG4gICAgICAgIGZzLndyaXRlRmlsZVN5bmModGhpcy5wa2dQYXRoLCB0aGlzLnBrZyk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEV4dHJhY3QgdmVyc2lvbnMgZm9yIGEgbGlzdCBvZiBtb2R1bGVzXG4gICAgICovXG4gICAgcHVibGljIGdldFZlcnNpb25zKG1vZHVsZXM6IHN0cmluZ1tdKToge1xuICAgICAgICBba2V5OiBzdHJpbmddOiBzdHJpbmc7XG4gICAgfSB7XG4gICAgICAgIGNvbnN0IGRlcGVuZGVuY2llczoge1xuICAgICAgICAgICAgW2tleTogc3RyaW5nXTogc3RyaW5nO1xuICAgICAgICB9ID0ge307XG4gICAgICAgIGNvbnN0IGFsbERlcGVuZGVuY2llcyA9IHtcbiAgICAgICAgICAgIC4uLnRoaXMucGtnSnNvbi5kZXBlbmRlbmNpZXMgPz8ge30sXG4gICAgICAgICAgICAuLi50aGlzLnBrZ0pzb24uZGV2RGVwZW5kZW5jaWVzID8/IHt9LFxuICAgICAgICAgICAgLi4udGhpcy5wa2dKc29uLnBlZXJEZXBlbmRlbmNpZXMgPz8ge30sXG4gICAgICAgIH07XG4gICAgICAgIGZvciAoY29uc3QgbW9kIG9mIG1vZHVsZXMpIHtcbiAgICAgICAgICAgIGlmICghYWxsRGVwZW5kZW5jaWVzW21vZF0pIHtcbiAgICAgICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYENhbm5vdCBleHRyYWN0IHZlcnNpb24gZm9yICR7bW9kfSBpbiBwYWNrYWdlLmpzb25gKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIGRlcGVuZGVuY2llc1ttb2RdID0gYWxsRGVwZW5kZW5jaWVzW21vZF07XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIGRlcGVuZGVuY2llcztcbiAgICB9XG59XG5pbnRlcmZhY2UgUGFja2FnZUpzb24ge1xuICAgIGRlcGVuZGVuY2llcz86IHtcbiAgICAgICAgW2tleTogc3RyaW5nXTogc3RyaW5nO1xuICAgIH07XG4gICAgZGV2RGVwZW5kZW5jaWVzPzoge1xuICAgICAgICBba2V5OiBzdHJpbmddOiBzdHJpbmc7XG4gICAgfTtcbiAgICBwZWVyRGVwZW5kZW5jaWVzPzoge1xuICAgICAgICBba2V5OiBzdHJpbmddOiBzdHJpbmc7XG4gICAgfTtcbn1cbiJdfQ==