"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.exec = exports.findUp = exports.nodeMajorVersion = exports.parseStackTrace = void 0;
const child_process_1 = require("child_process");
const fs = require("fs");
const path = require("path");
// From https://github.com/errwischt/stacktrace-parser/blob/master/src/stack-trace-parser.js
const STACK_RE = /^\s*at (?:((?:\[object object\])?[^\\/]+(?: \[as \S+\])?) )?\(?(.*?):(\d+)(?::(\d+))?\)?\s*$/i;
/**
 * Parses the stack trace of an error
 */
function parseStackTrace(error) {
    const err = error || new Error();
    if (!err.stack) {
        return [];
    }
    const lines = err.stack.split('\n');
    const stackTrace = [];
    for (const line of lines) {
        const results = STACK_RE.exec(line);
        if (results) {
            stackTrace.push({
                file: results[2],
                methodName: results[1],
                lineNumber: parseInt(results[3], 10),
                column: parseInt(results[4], 10),
            });
        }
    }
    return stackTrace;
}
exports.parseStackTrace = parseStackTrace;
/**
 * Returns the major version of node installation
 */
function nodeMajorVersion() {
    return parseInt(process.versions.node.split('.')[0], 10);
}
exports.nodeMajorVersion = nodeMajorVersion;
/**
 * Find a file by walking up parent directories
 */
function findUp(name, directory = process.cwd()) {
    const absoluteDirectory = path.resolve(directory);
    if (fs.existsSync(path.join(directory, name))) {
        return directory;
    }
    const { root } = path.parse(absoluteDirectory);
    if (absoluteDirectory === root) {
        return undefined;
    }
    return findUp(name, path.dirname(absoluteDirectory));
}
exports.findUp = findUp;
/**
 * Spawn sync with error handling
 */
function exec(cmd, args, options) {
    var _a, _b;
    const proc = child_process_1.spawnSync(cmd, args, options);
    if (proc.error) {
        throw proc.error;
    }
    if (proc.status !== 0) {
        if (proc.stdout || proc.stderr) {
            throw new Error(`[Status ${proc.status}] stdout: ${(_a = proc.stdout) === null || _a === void 0 ? void 0 : _a.toString().trim()}\n\n\nstderr: ${(_b = proc.stderr) === null || _b === void 0 ? void 0 : _b.toString().trim()}`);
        }
        throw new Error(`${cmd} exited with status ${proc.status}`);
    }
    return proc;
}
exports.exec = exec;
//# sourceMappingURL=data:application/json;base64,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