"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LayerVersion = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const lambda_generated_1 = require("./lambda.generated");
const runtime_1 = require("./runtime");
/**
 * A reference to a Lambda Layer version.
 */
class LayerVersionBase extends core_1.Resource {
    addPermission(id, permission) {
        if (permission.organizationId != null && permission.accountId !== '*') {
            throw new Error(`OrganizationId can only be specified if AwsAccountId is '*', but it is ${permission.accountId}`);
        }
        new lambda_generated_1.CfnLayerVersionPermission(this, id, {
            action: 'lambda:GetLayerVersion',
            layerVersionArn: this.layerVersionArn,
            principal: permission.accountId,
            organizationId: permission.organizationId,
        });
    }
}
/**
 * Defines a new Lambda Layer version.
 */
class LayerVersion extends LayerVersionBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.layerVersionName,
        });
        if (props.compatibleRuntimes && props.compatibleRuntimes.length === 0) {
            throw new Error('Attempted to define a Lambda layer that supports no runtime!');
        }
        if (props.code.isInline) {
            throw new Error('Lambda layers cannot be created from inline code');
        }
        // Allow usage of the code in this context...
        const code = props.code.bind(this);
        if (code.inlineCode) {
            throw new Error('Inline code is not supported for AWS Lambda layers');
        }
        if (!code.s3Location) {
            throw new Error('Code must define an S3 location');
        }
        const resource = new lambda_generated_1.CfnLayerVersion(this, 'Resource', {
            compatibleRuntimes: props.compatibleRuntimes && props.compatibleRuntimes.map(r => r.name),
            content: {
                s3Bucket: code.s3Location.bucketName,
                s3Key: code.s3Location.objectKey,
                s3ObjectVersion: code.s3Location.objectVersion,
            },
            description: props.description,
            layerName: this.physicalName,
            licenseInfo: props.license,
        });
        props.code.bindToResource(resource, {
            resourceProperty: 'Content',
        });
        this.layerVersionArn = resource.ref;
        this.compatibleRuntimes = props.compatibleRuntimes;
    }
    /**
     * Imports a layer version by ARN. Assumes it is compatible with all Lambda runtimes.
     */
    static fromLayerVersionArn(scope, id, layerVersionArn) {
        return LayerVersion.fromLayerVersionAttributes(scope, id, {
            layerVersionArn,
            compatibleRuntimes: runtime_1.Runtime.ALL,
        });
    }
    /**
     * Imports a Layer that has been defined externally.
     *
     * @param scope the parent Construct that will use the imported layer.
     * @param id    the id of the imported layer in the construct tree.
     * @param attrs the properties of the imported layer.
     */
    static fromLayerVersionAttributes(scope, id, attrs) {
        if (attrs.compatibleRuntimes && attrs.compatibleRuntimes.length === 0) {
            throw new Error('Attempted to import a Lambda layer that supports no runtime!');
        }
        class Import extends LayerVersionBase {
            constructor() {
                super(...arguments);
                this.layerVersionArn = attrs.layerVersionArn;
                this.compatibleRuntimes = attrs.compatibleRuntimes;
            }
        }
        return new Import(scope, id);
    }
}
exports.LayerVersion = LayerVersion;
//# sourceMappingURL=data:application/json;base64,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