"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Runtime = exports.RuntimeFamily = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
var RuntimeFamily;
(function (RuntimeFamily) {
    RuntimeFamily[RuntimeFamily["NODEJS"] = 0] = "NODEJS";
    RuntimeFamily[RuntimeFamily["JAVA"] = 1] = "JAVA";
    RuntimeFamily[RuntimeFamily["PYTHON"] = 2] = "PYTHON";
    RuntimeFamily[RuntimeFamily["DOTNET_CORE"] = 3] = "DOTNET_CORE";
    RuntimeFamily[RuntimeFamily["GO"] = 4] = "GO";
    RuntimeFamily[RuntimeFamily["RUBY"] = 5] = "RUBY";
    RuntimeFamily[RuntimeFamily["OTHER"] = 6] = "OTHER";
})(RuntimeFamily = exports.RuntimeFamily || (exports.RuntimeFamily = {}));
/**
 * Lambda function runtime environment.
 *
 * If you need to use a runtime name that doesn't exist as a static member, you
 * can instantiate a `Runtime` object, e.g: `new Runtime('nodejs99.99')`.
 */
class Runtime {
    constructor(name, family, props = {}) {
        var _a;
        this.name = name;
        this.supportsInlineCode = !!props.supportsInlineCode;
        this.family = family;
        const imageName = (_a = props.bundlingDockerImage) !== null && _a !== void 0 ? _a : `amazon/aws-sam-cli-build-image-${name}`;
        this.bundlingDockerImage = core_1.BundlingDockerImage.fromRegistry(imageName);
        Runtime.ALL.push(this);
    }
    toString() {
        return this.name;
    }
    runtimeEquals(other) {
        return other.name === this.name &&
            other.family === this.family &&
            other.supportsInlineCode === this.supportsInlineCode;
    }
}
exports.Runtime = Runtime;
/** A list of all known `Runtime`'s. */
Runtime.ALL = new Array();
/**
 * The NodeJS runtime (nodejs)
 *
 * @deprecated Use {@link NODEJS_10_X}
 */
Runtime.NODEJS = new Runtime('nodejs', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 4.3 runtime (nodejs4.3)
 *
 * @deprecated Use {@link NODEJS_10_X}
 */
Runtime.NODEJS_4_3 = new Runtime('nodejs4.3', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 6.10 runtime (nodejs6.10)
 *
 * @deprecated Use {@link NODEJS_10_X}
 */
Runtime.NODEJS_6_10 = new Runtime('nodejs6.10', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 8.10 runtime (nodejs8.10)
 *
 * @deprecated Use {@link NODEJS_10_X}
 */
Runtime.NODEJS_8_10 = new Runtime('nodejs8.10', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 10.x runtime (nodejs10.x)
 */
Runtime.NODEJS_10_X = new Runtime('nodejs10.x', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 12.x runtime (nodejs12.x)
 */
Runtime.NODEJS_12_X = new Runtime('nodejs12.x', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The Python 2.7 runtime (python2.7)
 */
Runtime.PYTHON_2_7 = new Runtime('python2.7', RuntimeFamily.PYTHON, { supportsInlineCode: true });
/**
 * The Python 3.6 runtime (python3.6)
 */
Runtime.PYTHON_3_6 = new Runtime('python3.6', RuntimeFamily.PYTHON, { supportsInlineCode: true });
/**
 * The Python 3.7 runtime (python3.7)
 */
Runtime.PYTHON_3_7 = new Runtime('python3.7', RuntimeFamily.PYTHON, { supportsInlineCode: true });
/**
 * The Python 3.8 runtime (python3.8)
 */
Runtime.PYTHON_3_8 = new Runtime('python3.8', RuntimeFamily.PYTHON);
/**
 * The Java 8 runtime (java8)
 */
Runtime.JAVA_8 = new Runtime('java8', RuntimeFamily.JAVA);
/**
 * The Java 8 Corretto runtime (java8.al2)
 */
Runtime.JAVA_8_CORRETTO = new Runtime('java8.al2', RuntimeFamily.JAVA);
/**
 * The Java 11 runtime (java11)
 */
Runtime.JAVA_11 = new Runtime('java11', RuntimeFamily.JAVA);
/**
 * The .NET Core 1.0 runtime (dotnetcore1.0)
 *
 * @deprecated Use {@link DOTNET_CORE_2_1}
 */
Runtime.DOTNET_CORE_1 = new Runtime('dotnetcore1.0', RuntimeFamily.DOTNET_CORE);
/**
 * The .NET Core 2.0 runtime (dotnetcore2.0)
 *
 * @deprecated Use {@link DOTNET_CORE_2_1}
 */
Runtime.DOTNET_CORE_2 = new Runtime('dotnetcore2.0', RuntimeFamily.DOTNET_CORE);
/**
 * The .NET Core 2.1 runtime (dotnetcore2.1)
 */
Runtime.DOTNET_CORE_2_1 = new Runtime('dotnetcore2.1', RuntimeFamily.DOTNET_CORE, {
    bundlingDockerImage: 'lambci/lambda:build-dotnetcore2.1',
});
/**
 * The .NET Core 3.1 runtime (dotnetcore3.1)
 */
Runtime.DOTNET_CORE_3_1 = new Runtime('dotnetcore3.1', RuntimeFamily.DOTNET_CORE, {
    bundlingDockerImage: 'lambci/lambda:build-dotnetcore3.1',
});
/**
 * The Go 1.x runtime (go1.x)
 */
Runtime.GO_1_X = new Runtime('go1.x', RuntimeFamily.GO, {
    bundlingDockerImage: 'lambci/lambda:build-go1.x',
});
/**
 * The Ruby 2.5 runtime (ruby2.5)
 */
Runtime.RUBY_2_5 = new Runtime('ruby2.5', RuntimeFamily.RUBY);
/**
 * The Ruby 2.7 runtime (ruby2.7)
 */
Runtime.RUBY_2_7 = new Runtime('ruby2.7', RuntimeFamily.RUBY);
/**
 * The custom provided runtime (provided)
 */
Runtime.PROVIDED = new Runtime('provided', RuntimeFamily.OTHER);
/**
 * The custom provided runtime (provided)
 */
Runtime.PROVIDED_AL2 = new Runtime('provided.al2', RuntimeFamily.OTHER);
//# sourceMappingURL=data:application/json;base64,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