"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.defaultDeletionProtection = exports.applyRemovalPolicy = exports.engineDescription = exports.setupS3ImportExport = exports.DEFAULT_PASSWORD_EXCLUDE_CHARS = void 0;
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * The default set of characters we exclude from generated passwords for database users.
 * It's a combination of characters that have a tendency to cause problems in shell scripts,
 * some engine-specific characters (for example, Oracle doesn't like '@' in its passwords),
 * and some that trip up other services, like DMS.
 *
 * This constant is private to the RDS module.
 */
exports.DEFAULT_PASSWORD_EXCLUDE_CHARS = " %+~`#$&*()|[]{}:;<>?!'/@\"\\";
/**
 * Validates the S3 import/export props and returns the import/export roles, if any.
 * If `combineRoles` is true, will reuse the import role for export (or vice versa) if possible.
 *
 * Notably, `combineRoles` is (by default) set to true for instances, but false for clusters.
 * This is because the `combineRoles` functionality is most applicable to instances and didn't exist
 * for the initial clusters implementation. To maintain backwards compatibility, it is set to false for clusters.
 */
function setupS3ImportExport(scope, props, combineRoles) {
    let s3ImportRole = props.s3ImportRole;
    let s3ExportRole = props.s3ExportRole;
    if (props.s3ImportBuckets && props.s3ImportBuckets.length > 0) {
        if (props.s3ImportRole) {
            throw new Error('Only one of s3ImportRole or s3ImportBuckets must be specified, not both.');
        }
        s3ImportRole = (combineRoles && s3ExportRole) ? s3ExportRole : new iam.Role(scope, 'S3ImportRole', {
            assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
        });
        for (const bucket of props.s3ImportBuckets) {
            bucket.grantRead(s3ImportRole);
        }
    }
    if (props.s3ExportBuckets && props.s3ExportBuckets.length > 0) {
        if (props.s3ExportRole) {
            throw new Error('Only one of s3ExportRole or s3ExportBuckets must be specified, not both.');
        }
        s3ExportRole = (combineRoles && s3ImportRole) ? s3ImportRole : new iam.Role(scope, 'S3ExportRole', {
            assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
        });
        for (const bucket of props.s3ExportBuckets) {
            bucket.grantReadWrite(s3ExportRole);
        }
    }
    return { s3ImportRole, s3ExportRole };
}
exports.setupS3ImportExport = setupS3ImportExport;
function engineDescription(engine) {
    var _a;
    return engine.engineType + (((_a = engine.engineVersion) === null || _a === void 0 ? void 0 : _a.fullVersion) ? `-${engine.engineVersion.fullVersion}` : '');
}
exports.engineDescription = engineDescription;
function applyRemovalPolicy(cfnDatabase, removalPolicy) {
    if (!removalPolicy) {
        // the default DeletionPolicy is 'Snapshot', which is fine,
        // but we should also make it 'Snapshot' for UpdateReplace policy
        cfnDatabase.cfnOptions.updateReplacePolicy = core_1.CfnDeletionPolicy.SNAPSHOT;
    }
    else {
        // just apply whatever removal policy the customer explicitly provided
        cfnDatabase.applyRemovalPolicy(removalPolicy);
    }
}
exports.applyRemovalPolicy = applyRemovalPolicy;
/**
 * By default, deletion protection is disabled.
 * Enable if explicitly provided or if the RemovalPolicy has been set to RETAIN
 */
function defaultDeletionProtection(deletionProtection, removalPolicy) {
    return deletionProtection !== undefined
        ? deletionProtection
        : (removalPolicy === core_1.RemovalPolicy.RETAIN ? true : undefined);
}
exports.defaultDeletionProtection = defaultDeletionProtection;
//# sourceMappingURL=data:application/json;base64,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