"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseProxy = exports.ProxyTarget = exports.SessionPinningFilter = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const util_1 = require("./private/util");
const rds_generated_1 = require("./rds.generated");
/**
 * SessionPinningFilter
 *
 * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/rds-proxy.html#rds-proxy-pinning
 */
class SessionPinningFilter {
    constructor(
    /**
     * Filter name
     */
    filterName) {
        this.filterName = filterName;
    }
    /**
     * custom filter
     */
    static of(filterName) {
        return new SessionPinningFilter(filterName);
    }
}
exports.SessionPinningFilter = SessionPinningFilter;
/**
 * You can opt out of session pinning for the following kinds of application statements:
 *
 * - Setting session variables and configuration settings.
 */
SessionPinningFilter.EXCLUDE_VARIABLE_SETS = new SessionPinningFilter('EXCLUDE_VARIABLE_SETS');
/**
 * Proxy target: Instance or Cluster
 *
 * A target group is a collection of databases that the proxy can connect to.
 * Currently, you can specify only one RDS DB instance or Aurora DB cluster.
 */
class ProxyTarget {
    constructor(dbInstance, dbCluster) {
        this.dbInstance = dbInstance;
        this.dbCluster = dbCluster;
    }
    /**
     * From instance
     *
     * @param instance RDS database instance
     */
    static fromInstance(instance) {
        return new ProxyTarget(instance, undefined);
    }
    /**
     * From cluster
     *
     * @param cluster RDS database cluster
     */
    static fromCluster(cluster) {
        return new ProxyTarget(undefined, cluster);
    }
    /**
     * Bind this target to the specified database proxy.
     */
    bind(_) {
        var _a, _b, _c, _d;
        const engine = (_b = (_a = this.dbInstance) === null || _a === void 0 ? void 0 : _a.engine) !== null && _b !== void 0 ? _b : (_c = this.dbCluster) === null || _c === void 0 ? void 0 : _c.engine;
        if (!engine) {
            const errorResource = (_d = this.dbCluster) !== null && _d !== void 0 ? _d : this.dbInstance;
            throw new Error(`Could not determine engine for proxy target '${errorResource === null || errorResource === void 0 ? void 0 : errorResource.node.path}'. ` +
                'Please provide it explicitly when importing the resource');
        }
        const engineFamily = engine.engineFamily;
        if (!engineFamily) {
            throw new Error(`Engine '${util_1.engineDescription(engine)}' does not support proxies`);
        }
        return {
            engineFamily,
            dbClusters: this.dbCluster ? [this.dbCluster] : undefined,
            dbInstances: this.dbInstance ? [this.dbInstance] : undefined,
        };
    }
}
exports.ProxyTarget = ProxyTarget;
/**
 * RDS Database Proxy
 *
 * @resource AWS::RDS::DBProxy
 */
class DatabaseProxy extends cdk.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id, { physicalName: props.dbProxyName || id });
        const role = props.role || new iam.Role(this, 'IAMRole', {
            assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
        });
        for (const secret of props.secrets) {
            secret.grantRead(role);
        }
        this.connections = new ec2.Connections({ securityGroups: props.securityGroups });
        const bindResult = props.proxyTarget.bind(this);
        if (props.secrets.length < 1) {
            throw new Error('One or more secrets are required.');
        }
        this.resource = new rds_generated_1.CfnDBProxy(this, 'Resource', {
            auth: props.secrets.map(_ => {
                return {
                    authScheme: 'SECRETS',
                    iamAuth: props.iamAuth ? 'REQUIRED' : 'DISABLED',
                    secretArn: _.secretArn,
                };
            }),
            dbProxyName: this.physicalName,
            debugLogging: props.debugLogging,
            engineFamily: bindResult.engineFamily,
            idleClientTimeout: (_a = props.idleClientTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
            requireTls: (_b = props.requireTLS) !== null && _b !== void 0 ? _b : true,
            roleArn: role.roleArn,
            vpcSecurityGroupIds: (_c = props.securityGroups) === null || _c === void 0 ? void 0 : _c.map(_ => _.securityGroupId),
            vpcSubnetIds: props.vpc.selectSubnets(props.vpcSubnets).subnetIds,
        });
        this.dbProxyName = this.resource.ref;
        this.dbProxyArn = this.resource.attrDbProxyArn;
        this.endpoint = this.resource.attrEndpoint;
        let dbInstanceIdentifiers;
        if (bindResult.dbInstances) {
            // support for only single instance
            dbInstanceIdentifiers = [bindResult.dbInstances[0].instanceIdentifier];
        }
        let dbClusterIdentifiers;
        if (bindResult.dbClusters) {
            dbClusterIdentifiers = bindResult.dbClusters.map((c) => c.clusterIdentifier);
        }
        if (!!dbInstanceIdentifiers && !!dbClusterIdentifiers) {
            throw new Error('Cannot specify both dbInstanceIdentifiers and dbClusterIdentifiers');
        }
        new rds_generated_1.CfnDBProxyTargetGroup(this, 'ProxyTargetGroup', {
            targetGroupName: 'default',
            dbProxyName: this.dbProxyName,
            dbInstanceIdentifiers,
            dbClusterIdentifiers,
            connectionPoolConfigurationInfo: toConnectionPoolConfigurationInfo(props),
        });
    }
    /**
     * Import an existing database proxy.
     */
    static fromDatabaseProxyAttributes(scope, id, attrs) {
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.dbProxyName = attrs.dbProxyName;
                this.dbProxyArn = attrs.dbProxyArn;
                this.endpoint = attrs.endpoint;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.dbProxyName,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_PROXY,
        };
    }
}
exports.DatabaseProxy = DatabaseProxy;
/**
 * ConnectionPoolConfiguration (L2 => L1)
 */
function toConnectionPoolConfigurationInfo(props) {
    var _a, _b;
    return {
        connectionBorrowTimeout: (_a = props.borrowTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
        initQuery: props.initQuery,
        maxConnectionsPercent: props.maxConnectionsPercent,
        maxIdleConnectionsPercent: props.maxIdleConnectionsPercent,
        sessionPinningFilters: (_b = props.sessionPinningFilters) === null || _b === void 0 ? void 0 : _b.map(_ => _.filterName),
    };
}
//# sourceMappingURL=data:application/json;base64,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