import * as ec2 from "../../aws-ec2";
import * as kms from "../../aws-kms";
import * as secretsmanager from "../../aws-secretsmanager";
import { Resource, Construct, Duration, RemovalPolicy, IResource } from "../../core";
import { IClusterEngine } from './cluster-engine';
import { Endpoint } from './endpoint';
import { IParameterGroup } from './parameter-group';
import { Credentials, RotationMultiUserOptions, RotationSingleUserOptions } from './props';
import { ISubnetGroup } from './subnet-group';
/**
  * Interface representing a serverless database cluster.
  *
  * @experimental
 */
export interface IServerlessCluster extends IResource, ec2.IConnectable, secretsmanager.ISecretAttachmentTarget {
    /**
     * Identifier of the cluster
     */
    readonly clusterIdentifier: string;
    /**
     * The endpoint to use for read/write operations
     * @attribute EndpointAddress,EndpointPort
     */
    readonly clusterEndpoint: Endpoint;
    /**
     * Endpoint to use for load-balanced read-only operations.
     * @attribute ReadEndpointAddress
     */
    readonly clusterReadEndpoint: Endpoint;
}
/**
 *  Properties to configure an Aurora Serverless Cluster
 *
 * @experimental
 */
export interface ServerlessClusterProps {
    /**
     * What kind of database to start
     */
    readonly engine: IClusterEngine;
    /**
     * Credentials for the administrative user
     *
     * @default - A username of 'admin' and SecretsManager-generated password
     */
    readonly credentials?: Credentials;
    /**
     * An optional identifier for the cluster
     *
     * @default - A name is automatically generated.
     */
    readonly clusterIdentifier?: string;
    /**
     * The number of days during which automatic DB snapshots are retained. Set
     * to zero to disable backups.
     *
     * @default Duration.days(1)
     */
    readonly backupRetention?: Duration;
    /**
     * Name of a database which is automatically created inside the cluster
     *
     * @default - Database is not created in cluster.
     */
    readonly defaultDatabaseName?: string;
    /**
     * Indicates whether the DB cluster should have deletion protection enabled.
     *
     * @default - true if removalPolicy is RETAIN, false otherwise
     */
    readonly deletionProtection?: boolean;
    /**
     * Whether to enable the HTTP endpoint for an Aurora Serverless database cluster.
     * The HTTP endpoint must be explicitly enabled to enable the Data API.
     *
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/data-api.html
      *
     * @default false
     */
    readonly enableHttpEndpoint?: boolean;
    /**
     * The VPC that this Aurora Serverless cluster has been created in.
     */
    readonly vpc: ec2.IVpc;
    /**
     * Where to place the instances within the VPC
     *
     * @default - the VPC default strategy if not specified.
     */
    readonly vpcSubnets?: ec2.SubnetSelection;
    /**
     * Scaling configuration of an Aurora Serverless database cluster.
     *
     * @default - Serverless cluster is automatically paused after 5 minutes of being idle.
     *   minimum capacity: 2 ACU
     *   maximum capacity: 16 ACU
     */
    readonly scaling?: ServerlessScalingOptions;
    /**
     * The removal policy to apply when the cluster and its instances are removed
     * from the stack or replaced during an update.
     *
     * @default - RemovalPolicy.SNAPSHOT (remove the cluster and instances, but retain a snapshot of the data)
     */
    readonly removalPolicy?: RemovalPolicy;
    /**
     * Security group.
     *
     * @default - a new security group is created.
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * The KMS key for storage encryption.
     *
     * @default - the default master key will be used for storage encryption
     */
    readonly storageEncryptionKey?: kms.IKey;
    /**
     * Additional parameters to pass to the database engine
     *
     * @default - no parameter group.
     */
    readonly parameterGroup?: IParameterGroup;
    /**
     * Existing subnet group for the cluster.
     *
     * @default - a new subnet group will be created.
     */
    readonly subnetGroup?: ISubnetGroup;
}
/**
 * Properties that describe an existing cluster instance
 *
 * @experimental
 */
export interface ServerlessClusterAttributes {
    /**
     * Identifier for the cluster
     */
    readonly clusterIdentifier: string;
    /**
     * The database port
     *
     * @default - none
     */
    readonly port?: number;
    /**
     * The security groups of the database cluster
     *
     * @default - no security groups
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * Cluster endpoint address
     *
     * @default - no endpoint address
     */
    readonly clusterEndpointAddress?: string;
    /**
     * Reader endpoint address
     *
     * @default - no reader address
     */
    readonly readerEndpointAddress?: string;
}
/**
 * Aurora capacity units (ACUs).
 * Each ACU is a combination of processing and memory capacity.
 *
 * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.setting-capacity.html
 * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.how-it-works.html#aurora-serverless.architecture
 *
 * @experimental
 */
export declare enum AuroraCapacityUnit {
    /** 1 Aurora Capacity Unit */
    ACU_1 = 1,
    /** 2 Aurora Capacity Units */
    ACU_2 = 2,
    /** 8 Aurora Capacity Units */
    ACU_8 = 8,
    /** 16 Aurora Capacity Units */
    ACU_16 = 16,
    /** 32 Aurora Capacity Units */
    ACU_32 = 32,
    /** 64 Aurora Capacity Units */
    ACU_64 = 64,
    /** 128 Aurora Capacity Units */
    ACU_128 = 128,
    /** 192 Aurora Capacity Units */
    ACU_192 = 192,
    /** 256 Aurora Capacity Units */
    ACU_256 = 256,
    /** 384 Aurora Capacity Units */
    ACU_384 = 384
}
/**
 * Options for configuring scaling on an Aurora Serverless cluster
 *
 * @experimental
 */
export interface ServerlessScalingOptions {
    /**
     * The minimum capacity for an Aurora Serverless database cluster.
     *
     * @default - determined by Aurora based on database engine
     */
    readonly minCapacity?: AuroraCapacityUnit;
    /**
     * The maximum capacity for an Aurora Serverless database cluster.
     *
     * @default - determined by Aurora based on database engine
     */
    readonly maxCapacity?: AuroraCapacityUnit;
    /**
     * The time before an Aurora Serverless database cluster is paused.
     * A database cluster can be paused only when it is idle (it has no connections).
     * Auto pause time must be between 5 minutes and 1 day.
     *
     * If a DB cluster is paused for more than seven days, the DB cluster might be
     * backed up with a snapshot. In this case, the DB cluster is restored when there
     * is a request to connect to it.
     *
     * Set to 0 to disable
     *
     * @default - automatic pause enabled after 5 minutes
     */
    readonly autoPause?: Duration;
}
/**
 * New or imported Serverless Cluster
 */
declare abstract class ServerlessClusterBase extends Resource implements IServerlessCluster {
    /**
     * Identifier of the cluster
     */
    abstract readonly clusterIdentifier: string;
    /**
     * The endpoint to use for read/write operations
     */
    abstract readonly clusterEndpoint: Endpoint;
    /**
     * The endpoint to use for read/write operations
     */
    abstract readonly clusterReadEndpoint: Endpoint;
    /**
     * Access to the network connections
     */
    abstract readonly connections: ec2.Connections;
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget(): secretsmanager.SecretAttachmentTargetProps;
}
/**
 * Create an Aurora Serverless Cluster
 *
 * @resource AWS::RDS::DBCluster
 *
 * @experimental
 */
export declare class ServerlessCluster extends ServerlessClusterBase {
    /**
     * Import an existing DatabaseCluster from properties
     */
    static fromServerlessClusterAttributes(scope: Construct, id: string, attrs: ServerlessClusterAttributes): IServerlessCluster;
    readonly clusterIdentifier: string;
    readonly clusterEndpoint: Endpoint;
    readonly clusterReadEndpoint: Endpoint;
    readonly connections: ec2.Connections;
    /**
     * The secret attached to this cluster
     */
    private readonly secret?;
    private readonly subnetGroup;
    private readonly vpc;
    private readonly vpcSubnets?;
    private readonly singleUserRotationApplication;
    private readonly multiUserRotationApplication;
    constructor(scope: Construct, id: string, props: ServerlessClusterProps);
    /**
     * Adds the single user rotation of the master password to this cluster.
     */
    addRotationSingleUser(options?: RotationSingleUserOptions): secretsmanager.SecretRotation;
    /**
     * Adds the multi user rotation to this cluster.
     */
    addRotationMultiUser(id: string, options: RotationMultiUserOptions): secretsmanager.SecretRotation;
    private renderScalingConfiguration;
}
export {};
