"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerlessCluster = exports.AuroraCapacityUnit = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const database_secret_1 = require("./database-secret");
const endpoint_1 = require("./endpoint");
const util_1 = require("./private/util");
const props_1 = require("./props");
const rds_generated_1 = require("./rds.generated");
const subnet_group_1 = require("./subnet-group");
/**
 * Aurora capacity units (ACUs).
 * Each ACU is a combination of processing and memory capacity.
 *
 * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.setting-capacity.html
 * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.how-it-works.html#aurora-serverless.architecture
 *
 * @experimental
 */
var AuroraCapacityUnit;
(function (AuroraCapacityUnit) {
    /** 1 Aurora Capacity Unit */
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_1"] = 1] = "ACU_1";
    /** 2 Aurora Capacity Units */
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_2"] = 2] = "ACU_2";
    /** 8 Aurora Capacity Units */
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_8"] = 8] = "ACU_8";
    /** 16 Aurora Capacity Units */
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_16"] = 16] = "ACU_16";
    /** 32 Aurora Capacity Units */
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_32"] = 32] = "ACU_32";
    /** 64 Aurora Capacity Units */
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_64"] = 64] = "ACU_64";
    /** 128 Aurora Capacity Units */
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_128"] = 128] = "ACU_128";
    /** 192 Aurora Capacity Units */
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_192"] = 192] = "ACU_192";
    /** 256 Aurora Capacity Units */
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_256"] = 256] = "ACU_256";
    /** 384 Aurora Capacity Units */
    AuroraCapacityUnit[AuroraCapacityUnit["ACU_384"] = 384] = "ACU_384";
})(AuroraCapacityUnit = exports.AuroraCapacityUnit || (exports.AuroraCapacityUnit = {}));
/**
 * New or imported Serverless Cluster
 */
class ServerlessClusterBase extends core_1.Resource {
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.clusterIdentifier,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_CLUSTER,
        };
    }
}
/**
 * Create an Aurora Serverless Cluster
 *
 * @resource AWS::RDS::DBCluster
 *
 * @experimental
 */
class ServerlessCluster extends ServerlessClusterBase {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j;
        super(scope, id);
        this.vpc = props.vpc;
        this.vpcSubnets = props.vpcSubnets;
        this.singleUserRotationApplication = props.engine.singleUserRotationApplication;
        this.multiUserRotationApplication = props.engine.multiUserRotationApplication;
        const { subnetIds } = this.vpc.selectSubnets(this.vpcSubnets);
        // Cannot test whether the subnets are in different AZs, but at least we can test the amount.
        if (subnetIds.length < 2) {
            core_1.Annotations.of(this).addError(`Cluster requires at least 2 subnets, got ${subnetIds.length}`);
        }
        this.subnetGroup = (_a = props.subnetGroup) !== null && _a !== void 0 ? _a : new subnet_group_1.SubnetGroup(this, 'Subnets', {
            description: `Subnets for ${id} database`,
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets,
            removalPolicy: props.removalPolicy === core_1.RemovalPolicy.RETAIN ? props.removalPolicy : undefined,
        });
        let credentials = (_b = props.credentials) !== null && _b !== void 0 ? _b : props_1.Credentials.fromUsername((_c = props.engine.defaultUsername) !== null && _c !== void 0 ? _c : 'admin');
        if (!credentials.secret && !credentials.password) {
            credentials = props_1.Credentials.fromSecret(new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: credentials.username,
                encryptionKey: credentials.encryptionKey,
                excludeCharacters: credentials.excludeCharacters,
            }));
        }
        const secret = credentials.secret;
        // bind the engine to the Cluster
        const clusterEngineBindConfig = props.engine.bindToCluster(this, {
            parameterGroup: props.parameterGroup,
        });
        const clusterParameterGroup = (_d = props.parameterGroup) !== null && _d !== void 0 ? _d : clusterEngineBindConfig.parameterGroup;
        const clusterParameterGroupConfig = clusterParameterGroup === null || clusterParameterGroup === void 0 ? void 0 : clusterParameterGroup.bindToCluster({});
        const securityGroups = (_e = props.securityGroups) !== null && _e !== void 0 ? _e : [
            new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: 'RDS security group',
                vpc: this.vpc,
            }),
        ];
        const cluster = new rds_generated_1.CfnDBCluster(this, 'Resource', {
            backupRetentionPeriod: (_f = props.backupRetention) === null || _f === void 0 ? void 0 : _f.toDays(),
            databaseName: props.defaultDatabaseName,
            dbClusterIdentifier: props.clusterIdentifier,
            dbClusterParameterGroupName: clusterParameterGroupConfig === null || clusterParameterGroupConfig === void 0 ? void 0 : clusterParameterGroupConfig.parameterGroupName,
            dbSubnetGroupName: this.subnetGroup.subnetGroupName,
            deletionProtection: util_1.defaultDeletionProtection(props.deletionProtection, props.removalPolicy),
            engine: props.engine.engineType,
            engineVersion: (_g = props.engine.engineVersion) === null || _g === void 0 ? void 0 : _g.fullVersion,
            engineMode: 'serverless',
            enableHttpEndpoint: props.enableHttpEndpoint,
            kmsKeyId: (_h = props.storageEncryptionKey) === null || _h === void 0 ? void 0 : _h.keyArn,
            masterUsername: credentials.username,
            masterUserPassword: (_j = credentials.password) === null || _j === void 0 ? void 0 : _j.toString(),
            scalingConfiguration: props.scaling ? this.renderScalingConfiguration(props.scaling) : undefined,
            storageEncrypted: true,
            vpcSecurityGroupIds: securityGroups.map(sg => sg.securityGroupId),
        });
        this.clusterIdentifier = cluster.ref;
        // create a number token that represents the port of the cluster
        const portAttribute = core_1.Token.asNumber(cluster.attrEndpointPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpointAddress, portAttribute);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(cluster.attrReadEndpointAddress, portAttribute);
        this.connections = new ec2.Connections({
            securityGroups,
            defaultPort: ec2.Port.tcp(this.clusterEndpoint.port),
        });
        util_1.applyRemovalPolicy(cluster, props.removalPolicy);
        if (secret) {
            this.secret = secret.attach(this);
        }
    }
    /**
     * Import an existing DatabaseCluster from properties
     */
    static fromServerlessClusterAttributes(scope, id, attrs) {
        return new ImportedServerlessCluster(scope, id, attrs);
    }
    /**
     * Adds the single user rotation of the master password to this cluster.
     */
    addRotationSingleUser(options = {}) {
        var _a;
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for a cluster without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this cluster.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
            ...options,
            excludeCharacters: (_a = options.excludeCharacters) !== null && _a !== void 0 ? _a : util_1.DEFAULT_PASSWORD_EXCLUDE_CHARS,
        });
    }
    /**
     * Adds the multi user rotation to this cluster.
     */
    addRotationMultiUser(id, options) {
        var _a;
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for a cluster without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            ...options,
            excludeCharacters: (_a = options.excludeCharacters) !== null && _a !== void 0 ? _a : util_1.DEFAULT_PASSWORD_EXCLUDE_CHARS,
            masterSecret: this.secret,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    renderScalingConfiguration(options) {
        var _a;
        const minCapacity = options.minCapacity;
        const maxCapacity = options.maxCapacity;
        if (minCapacity && maxCapacity && minCapacity > maxCapacity) {
            throw new Error('maximum capacity must be greater than or equal to minimum capacity.');
        }
        const secondsToAutoPause = (_a = options.autoPause) === null || _a === void 0 ? void 0 : _a.toSeconds();
        if (secondsToAutoPause && (secondsToAutoPause < 300 || secondsToAutoPause > 86400)) {
            throw new Error('auto pause time must be between 5 minutes and 1 day.');
        }
        return {
            autoPause: (secondsToAutoPause === 0) ? false : true,
            minCapacity: options.minCapacity,
            maxCapacity: options.maxCapacity,
            secondsUntilAutoPause: (secondsToAutoPause === 0) ? undefined : secondsToAutoPause,
        };
    }
}
exports.ServerlessCluster = ServerlessCluster;
/**
 * Represents an imported database cluster.
 */
class ImportedServerlessCluster extends ServerlessClusterBase {
    constructor(scope, id, attrs) {
        super(scope, id);
        this.clusterIdentifier = attrs.clusterIdentifier;
        const defaultPort = attrs.port ? ec2.Port.tcp(attrs.port) : undefined;
        this.connections = new ec2.Connections({
            securityGroups: attrs.securityGroups,
            defaultPort,
        });
        this._clusterEndpoint = (attrs.clusterEndpointAddress && attrs.port) ? new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.port) : undefined;
        this._clusterReadEndpoint = (attrs.readerEndpointAddress && attrs.port) ? new endpoint_1.Endpoint(attrs.readerEndpointAddress, attrs.port) : undefined;
    }
    get clusterEndpoint() {
        if (!this._clusterEndpoint) {
            throw new Error('Cannot access `clusterEndpoint` of an imported cluster without an endpoint address and port');
        }
        return this._clusterEndpoint;
    }
    get clusterReadEndpoint() {
        if (!this._clusterReadEndpoint) {
            throw new Error('Cannot access `clusterReadEndpoint` of an imported cluster without a readerEndpointAddress and port');
        }
        return this._clusterReadEndpoint;
    }
}
//# sourceMappingURL=data:application/json;base64,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