"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PrivateHostedZone = exports.PublicHostedZone = exports.HostedZone = void 0;
const cxschema = require("../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const record_set_1 = require("./record-set");
const route53_generated_1 = require("./route53.generated");
const util_1 = require("./util");
/**
 * Container for records, and records contain information about how to route traffic for a
 * specific domain, such as example.com and its subdomains (acme.example.com, zenith.example.com)
 */
class HostedZone extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * VPCs to which this hosted zone will be added
         */
        this.vpcs = new Array();
        util_1.validateZoneName(props.zoneName);
        const resource = new route53_generated_1.CfnHostedZone(this, 'Resource', {
            name: props.zoneName + '.',
            hostedZoneConfig: props.comment ? { comment: props.comment } : undefined,
            queryLoggingConfig: props.queryLogsLogGroupArn ? { cloudWatchLogsLogGroupArn: props.queryLogsLogGroupArn } : undefined,
            vpcs: core_1.Lazy.anyValue({ produce: () => this.vpcs.length === 0 ? undefined : this.vpcs }),
        });
        this.hostedZoneId = resource.ref;
        this.hostedZoneNameServers = resource.attrNameServers;
        this.zoneName = props.zoneName;
        for (const vpc of props.vpcs || []) {
            this.addVpc(vpc);
        }
    }
    get hostedZoneArn() {
        return util_1.makeHostedZoneArn(this, this.hostedZoneId);
    }
    /**
     * Import a Route 53 hosted zone defined either outside the CDK, or in a different CDK stack
     *
     * Use when hosted zone ID is known. Hosted zone name becomes unavailable through this query.
     *
     * @param scope the parent Construct for this Construct
     * @param id  the logical name of this Construct
     * @param hostedZoneId the ID of the hosted zone to import
     */
    static fromHostedZoneId(scope, id, hostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = hostedZoneId;
            }
            get zoneName() {
                throw new Error('HostedZone.fromHostedZoneId doesn\'t support "zoneName"');
            }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports a hosted zone from another stack.
     *
     * Use when both hosted zone ID and hosted zone name are known.
     *
     * @param scope the parent Construct for this Construct
     * @param id  the logical name of this Construct
     * @param attrs the HostedZoneAttributes (hosted zone ID and hosted zone name)
     */
    static fromHostedZoneAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = attrs.hostedZoneId;
                this.zoneName = attrs.zoneName;
            }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Lookup a hosted zone in the current account/region based on query parameters.
     * Requires environment, you must specify env for the stack.
     *
     * Use to easily query hosted zones.
     *
     * @see https://docs.aws.amazon.com/cdk/latest/guide/environments.html
     */
    static fromLookup(scope, id, query) {
        const DEFAULT_HOSTED_ZONE = {
            Id: 'DUMMY',
            Name: query.domainName,
        };
        const response = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.HOSTED_ZONE_PROVIDER,
            dummyValue: DEFAULT_HOSTED_ZONE,
            props: query,
        }).value;
        // CDK handles the '.' at the end, so remove it here
        if (response.Name.endsWith('.')) {
            response.Name = response.Name.substring(0, response.Name.length - 1);
        }
        response.Id = response.Id.replace('/hostedzone/', '');
        return HostedZone.fromHostedZoneAttributes(scope, id, {
            hostedZoneId: response.Id,
            zoneName: response.Name,
        });
    }
    /**
     * Add another VPC to this private hosted zone.
     *
     * @param vpc the other VPC to add.
     */
    addVpc(vpc) {
        this.vpcs.push({ vpcId: vpc.vpcId, vpcRegion: core_1.Stack.of(vpc).region });
    }
}
exports.HostedZone = HostedZone;
/**
 * Create a Route53 public hosted zone.
 *
 * @resource AWS::Route53::HostedZone
 */
class PublicHostedZone extends HostedZone {
    /**
     * Import a Route 53 public hosted zone defined either outside the CDK, or in a different CDK stack
     *
     * @param scope the parent Construct for this Construct
     * @param id the logical name of this Construct
     * @param publicHostedZoneId the ID of the public hosted zone to import
     */
    static fromPublicHostedZoneId(scope, id, publicHostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = publicHostedZoneId;
            }
            get zoneName() { throw new Error('cannot retrieve "zoneName" from an an imported hosted zone'); }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, props);
        if (props.caaAmazon) {
            new record_set_1.CaaAmazonRecord(this, 'CaaAmazon', {
                zone: this,
            });
        }
    }
    addVpc(_vpc) {
        throw new Error('Cannot associate public hosted zones with a VPC');
    }
    /**
     * Adds a delegation from this zone to a designated zone.
     *
     * @param delegate the zone being delegated to.
     * @param opts     options for creating the DNS record, if any.
     */
    addDelegation(delegate, opts = {}) {
        new record_set_1.ZoneDelegationRecord(this, `${this.zoneName} -> ${delegate.zoneName}`, {
            zone: this,
            recordName: delegate.zoneName,
            nameServers: delegate.hostedZoneNameServers,
            comment: opts.comment,
            ttl: opts.ttl,
        });
    }
}
exports.PublicHostedZone = PublicHostedZone;
/**
 * Create a Route53 private hosted zone for use in one or more VPCs.
 *
 * Note that `enableDnsHostnames` and `enableDnsSupport` must have been enabled
 * for the VPC you're configuring for private hosted zones.
 *
 * @resource AWS::Route53::HostedZone
 */
class PrivateHostedZone extends HostedZone {
    /**
     * Import a Route 53 private hosted zone defined either outside the CDK, or in a different CDK stack
     *
     * @param scope the parent Construct for this Construct
     * @param id the logical name of this Construct
     * @param privateHostedZoneId the ID of the private hosted zone to import
     */
    static fromPrivateHostedZoneId(scope, id, privateHostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = privateHostedZoneId;
            }
            get zoneName() { throw new Error('cannot retrieve "zoneName" from an an imported hosted zone'); }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, props);
        this.addVpc(props.vpc);
    }
}
exports.PrivateHostedZone = PrivateHostedZone;
//# sourceMappingURL=data:application/json;base64,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