"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Source = void 0;
const s3_assets = require("../../aws-s3-assets"); // Automatically re-written from '@aws-cdk/aws-s3-assets'
/**
 * Specifies bucket deployment source.
 *
 * Usage:
 *
 *     Source.bucket(bucket, key)
 *     Source.asset('/local/path/to/directory')
 *     Source.asset('/local/path/to/a/file.zip')
 *
 */
class Source {
    /**
     * Uses a .zip file stored in an S3 bucket as the source for the destination bucket contents.
     * @param bucket The S3 Bucket
     * @param zipObjectKey The S3 object key of the zip file with contents
     */
    static bucket(bucket, zipObjectKey) {
        return {
            bind: (_, context) => {
                if (!context) {
                    throw new Error('To use a Source.bucket(), context must be provided');
                }
                bucket.grantRead(context.handlerRole);
                return { bucket, zipObjectKey };
            },
        };
    }
    /**
     * Uses a local asset as the deployment source.
     * @param path The path to a local .zip file or a directory
     */
    static asset(path, options) {
        return {
            bind(scope, context) {
                if (!context) {
                    throw new Error('To use a Source.asset(), context must be provided');
                }
                let id = 1;
                while (scope.node.tryFindChild(`Asset${id}`)) {
                    id++;
                }
                const asset = new s3_assets.Asset(scope, `Asset${id}`, {
                    path,
                    ...options,
                });
                if (!asset.isZipArchive) {
                    throw new Error('Asset path must be either a .zip file or a directory');
                }
                asset.grantRead(context.handlerRole);
                return {
                    bucket: asset.bucket,
                    zipObjectKey: asset.s3ObjectKey,
                };
            },
        };
    }
    constructor() { }
}
exports.Source = Source;
//# sourceMappingURL=data:application/json;base64,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