"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Lambda = exports.LambdaInvocationType = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * The type of invocation to use for a Lambda Action.
 */
var LambdaInvocationType;
(function (LambdaInvocationType) {
    /**
     * The function will be invoked asynchronously.
     */
    LambdaInvocationType["EVENT"] = "Event";
    /**
     * The function will be invoked sychronously. Use RequestResponse only when
     * you want to make a mail flow decision, such as whether to stop the receipt
     * rule or the receipt rule set.
     */
    LambdaInvocationType["REQUEST_RESPONSE"] = "RequestResponse";
})(LambdaInvocationType = exports.LambdaInvocationType || (exports.LambdaInvocationType = {}));
/**
 * Calls an AWS Lambda function, and optionally, publishes a notification to
 * Amazon SNS.
 */
class Lambda {
    constructor(props) {
        this.props = props;
    }
    bind(rule) {
        // Allow SES to invoke Lambda function
        // See https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html#receiving-email-permissions-lambda
        const permissionId = 'AllowSes';
        if (!this.props.function.permissionsNode.tryFindChild(permissionId)) {
            this.props.function.addPermission(permissionId, {
                action: 'lambda:InvokeFunction',
                principal: new iam.ServicePrincipal('ses.amazonaws.com'),
                sourceAccount: cdk.Aws.ACCOUNT_ID,
            });
        }
        // Ensure permission is deployed before rule
        const permission = this.props.function.permissionsNode.tryFindChild(permissionId);
        if (permission) { // The Lambda could be imported
            rule.node.addDependency(permission);
        }
        else {
            // eslint-disable-next-line max-len
            cdk.Annotations.of(rule).addWarning('This rule is using a Lambda action with an imported function. Ensure permission is given to SES to invoke that function.');
        }
        return {
            lambdaAction: {
                functionArn: this.props.function.functionArn,
                invocationType: this.props.invocationType,
                topicArn: this.props.topic ? this.props.topic.topicArn : undefined,
            },
        };
    }
}
exports.Lambda = Lambda;
//# sourceMappingURL=data:application/json;base64,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