"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3 = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Saves the received message to an Amazon S3 bucket and, optionally, publishes
 * a notification to Amazon SNS.
 */
class S3 {
    constructor(props) {
        this.props = props;
    }
    bind(rule) {
        // Allow SES to write to S3 bucket
        // See https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html#receiving-email-permissions-s3
        const keyPattern = this.props.objectKeyPrefix || '';
        const s3Statement = new iam.PolicyStatement({
            actions: ['s3:PutObject'],
            principals: [new iam.ServicePrincipal('ses.amazonaws.com')],
            resources: [this.props.bucket.arnForObjects(`${keyPattern}*`)],
            conditions: {
                StringEquals: {
                    'aws:Referer': cdk.Aws.ACCOUNT_ID,
                },
            },
        });
        this.props.bucket.addToResourcePolicy(s3Statement);
        const policy = this.props.bucket.node.tryFindChild('Policy');
        if (policy) { // The bucket could be imported
            rule.node.addDependency(policy);
        }
        else {
            cdk.Annotations.of(rule).addWarning('This rule is using a S3 action with an imported bucket. Ensure permission is given to SES to write to that bucket.');
        }
        // Allow SES to use KMS master key
        // See https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html#receiving-email-permissions-kms
        if (this.props.kmsKey && !/alias\/aws\/ses$/.test(this.props.kmsKey.keyArn)) {
            const kmsStatement = new iam.PolicyStatement({
                actions: ['km:Encrypt', 'kms:GenerateDataKey'],
                principals: [new iam.ServicePrincipal('ses.amazonaws.com')],
                resources: ['*'],
                conditions: {
                    Null: {
                        'kms:EncryptionContext:aws:ses:rule-name': 'false',
                        'kms:EncryptionContext:aws:ses:message-id': 'false',
                    },
                    StringEquals: {
                        'kms:EncryptionContext:aws:ses:source-account': cdk.Aws.ACCOUNT_ID,
                    },
                },
            });
            this.props.kmsKey.addToResourcePolicy(kmsStatement);
        }
        return {
            s3Action: {
                bucketName: this.props.bucket.bucketName,
                kmsKeyArn: this.props.kmsKey ? this.props.kmsKey.keyArn : undefined,
                objectKeyPrefix: this.props.objectKeyPrefix,
                topicArn: this.props.topic ? this.props.topic.topicArn : undefined,
            },
        };
    }
}
exports.S3 = S3;
//# sourceMappingURL=data:application/json;base64,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