"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeBuildStartBuild = void 0;
const codebuild = require("../../../aws-codebuild"); // Automatically re-written from '@aws-cdk/aws-codebuild'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const task_utils_1 = require("../private/task-utils");
/**
 * Start a CodeBuild Build as a task
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-codebuild.html
 */
class CodeBuildStartBuild extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = (_a = props.integrationPattern) !== null && _a !== void 0 ? _a : sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, CodeBuildStartBuild.SUPPORTED_INTEGRATION_PATTERNS);
        this.taskMetrics = {
            metricPrefixSingular: 'CodeBuildProject',
            metricPrefixPlural: 'CodeBuildProjects',
            metricDimensions: {
                ProjectArn: this.props.project.projectArn,
            },
        };
        this.taskPolicies = this.configurePolicyStatements();
    }
    configurePolicyStatements() {
        let policyStatements = [
            new iam.PolicyStatement({
                resources: [this.props.project.projectArn],
                actions: [
                    'codebuild:StartBuild',
                    'codebuild:StopBuild',
                    'codebuild:BatchGetBuilds',
                    'codebuild:BatchGetReports',
                ],
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    cdk.Stack.of(this).formatArn({
                        service: 'events',
                        resource: 'rule/StepFunctionsGetEventForCodeBuildStartBuildRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
    /**
     * Provides the CodeBuild StartBuild service integration task configuration
     */
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('codebuild', 'startBuild', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                ProjectName: this.props.project.projectName,
                EnvironmentVariablesOverride: this.props.environmentVariablesOverride
                    ? this.serializeEnvVariables(this.props.environmentVariablesOverride)
                    : undefined,
            }),
        };
    }
    serializeEnvVariables(environmentVariables) {
        return Object.keys(environmentVariables).map(name => ({
            Name: name,
            Type: environmentVariables[name].type || codebuild.BuildEnvironmentVariableType.PLAINTEXT,
            Value: environmentVariables[name].value,
        }));
    }
}
exports.CodeBuildStartBuild = CodeBuildStartBuild;
CodeBuildStartBuild.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3RhcnQtYnVpbGQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJzdGFydC1idWlsZC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxvREFBb0QsQ0FBQyx5REFBeUQ7QUFDOUcsd0NBQXdDLENBQUMsbURBQW1EO0FBQzVGLGtEQUFrRCxDQUFDLDZEQUE2RDtBQUNoSCxxQ0FBcUMsQ0FBQyxnREFBZ0Q7QUFDdEYsc0RBQXlGO0FBa0J6Rjs7OztHQUlHO0FBQ0gsTUFBYSxtQkFBb0IsU0FBUSxHQUFHLENBQUMsYUFBYTtJQVF0RCxZQUFZLEtBQW9CLEVBQUUsRUFBVSxFQUFtQixLQUErQjs7UUFDMUYsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFEbUMsVUFBSyxHQUFMLEtBQUssQ0FBMEI7UUFFMUYsSUFBSSxDQUFDLGtCQUFrQixTQUFHLEtBQUssQ0FBQyxrQkFBa0IsbUNBQUksR0FBRyxDQUFDLGtCQUFrQixDQUFDLGdCQUFnQixDQUFDO1FBQzlGLHFDQUF3QixDQUFDLElBQUksQ0FBQyxrQkFBa0IsRUFBRSxtQkFBbUIsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBQ3RHLElBQUksQ0FBQyxXQUFXLEdBQUc7WUFDZixvQkFBb0IsRUFBRSxrQkFBa0I7WUFDeEMsa0JBQWtCLEVBQUUsbUJBQW1CO1lBQ3ZDLGdCQUFnQixFQUFFO2dCQUNkLFVBQVUsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxVQUFVO2FBQzVDO1NBQ0osQ0FBQztRQUNGLElBQUksQ0FBQyxZQUFZLEdBQUcsSUFBSSxDQUFDLHlCQUF5QixFQUFFLENBQUM7SUFDekQsQ0FBQztJQUNPLHlCQUF5QjtRQUM3QixJQUFJLGdCQUFnQixHQUFHO1lBQ25CLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztnQkFDcEIsU0FBUyxFQUFFLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDO2dCQUMxQyxPQUFPLEVBQUU7b0JBQ0wsc0JBQXNCO29CQUN0QixxQkFBcUI7b0JBQ3JCLDBCQUEwQjtvQkFDMUIsMkJBQTJCO2lCQUM5QjthQUNKLENBQUM7U0FDTCxDQUFDO1FBQ0YsSUFBSSxJQUFJLENBQUMsa0JBQWtCLEtBQUssR0FBRyxDQUFDLGtCQUFrQixDQUFDLE9BQU8sRUFBRTtZQUM1RCxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO2dCQUMxQyxPQUFPLEVBQUUsQ0FBQyxtQkFBbUIsRUFBRSxnQkFBZ0IsRUFBRSxxQkFBcUIsQ0FBQztnQkFDdkUsU0FBUyxFQUFFO29CQUNQLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFNBQVMsQ0FBQzt3QkFDekIsT0FBTyxFQUFFLFFBQVE7d0JBQ2pCLFFBQVEsRUFBRSxzREFBc0Q7cUJBQ25FLENBQUM7aUJBQ0w7YUFDSixDQUFDLENBQUMsQ0FBQztTQUNQO1FBQ0QsT0FBTyxnQkFBZ0IsQ0FBQztJQUM1QixDQUFDO0lBQ0Q7O09BRUc7SUFDSDs7T0FFRztJQUNPLFdBQVc7UUFDakIsT0FBTztZQUNILFFBQVEsRUFBRSxtQ0FBc0IsQ0FBQyxXQUFXLEVBQUUsWUFBWSxFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQztZQUNwRixVQUFVLEVBQUUsR0FBRyxDQUFDLFVBQVUsQ0FBQyxZQUFZLENBQUM7Z0JBQ3BDLFdBQVcsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxXQUFXO2dCQUMzQyw0QkFBNEIsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLDRCQUE0QjtvQkFDakUsQ0FBQyxDQUFDLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLDRCQUE0QixDQUFDO29CQUNyRSxDQUFDLENBQUMsU0FBUzthQUNsQixDQUFDO1NBQ0wsQ0FBQztJQUNOLENBQUM7SUFDTyxxQkFBcUIsQ0FBQyxvQkFFN0I7UUFDRyxPQUFPLE1BQU0sQ0FBQyxJQUFJLENBQUMsb0JBQW9CLENBQUMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1lBQ2xELElBQUksRUFBRSxJQUFJO1lBQ1YsSUFBSSxFQUFFLG9CQUFvQixDQUFDLElBQUksQ0FBQyxDQUFDLElBQUksSUFBSSxTQUFTLENBQUMsNEJBQTRCLENBQUMsU0FBUztZQUN6RixLQUFLLEVBQUUsb0JBQW9CLENBQUMsSUFBSSxDQUFDLENBQUMsS0FBSztTQUMxQyxDQUFDLENBQUMsQ0FBQztJQUNSLENBQUM7O0FBdkVMLGtEQXdFQztBQXZFMkIsa0RBQThCLEdBQTZCO0lBQy9FLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxnQkFBZ0I7SUFDdkMsR0FBRyxDQUFDLGtCQUFrQixDQUFDLE9BQU87Q0FDakMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNvZGVidWlsZCBmcm9tIFwiLi4vLi4vLi4vYXdzLWNvZGVidWlsZFwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWNvZGVidWlsZCdcbmltcG9ydCAqIGFzIGlhbSBmcm9tIFwiLi4vLi4vLi4vYXdzLWlhbVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSdcbmltcG9ydCAqIGFzIHNmbiBmcm9tIFwiLi4vLi4vLi4vYXdzLXN0ZXBmdW5jdGlvbnNcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1zdGVwZnVuY3Rpb25zJ1xuaW1wb3J0ICogYXMgY2RrIGZyb20gXCIuLi8uLi8uLi9jb3JlXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9jb3JlJ1xuaW1wb3J0IHsgaW50ZWdyYXRpb25SZXNvdXJjZUFybiwgdmFsaWRhdGVQYXR0ZXJuU3VwcG9ydGVkIH0gZnJvbSAnLi4vcHJpdmF0ZS90YXNrLXV0aWxzJztcbi8qKlxuICogUHJvcGVydGllcyBmb3IgQ29kZUJ1aWxkU3RhcnRCdWlsZFxuICovXG5leHBvcnQgaW50ZXJmYWNlIENvZGVCdWlsZFN0YXJ0QnVpbGRQcm9wcyBleHRlbmRzIHNmbi5UYXNrU3RhdGVCYXNlUHJvcHMge1xuICAgIC8qKlxuICAgICAqIENvZGVCdWlsZCBwcm9qZWN0IHRvIHN0YXJ0XG4gICAgICovXG4gICAgcmVhZG9ubHkgcHJvamVjdDogY29kZWJ1aWxkLklQcm9qZWN0O1xuICAgIC8qKlxuICAgICAqIEEgc2V0IG9mIGVudmlyb25tZW50IHZhcmlhYmxlcyB0byBiZSB1c2VkIGZvciB0aGlzIGJ1aWxkIG9ubHkuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIHRoZSBsYXRlc3QgZW52aXJvbm1lbnQgdmFyaWFibGVzIGFscmVhZHkgZGVmaW5lZCBpbiB0aGUgYnVpbGQgcHJvamVjdC5cbiAgICAgKi9cbiAgICByZWFkb25seSBlbnZpcm9ubWVudFZhcmlhYmxlc092ZXJyaWRlPzoge1xuICAgICAgICBbbmFtZTogc3RyaW5nXTogY29kZWJ1aWxkLkJ1aWxkRW52aXJvbm1lbnRWYXJpYWJsZTtcbiAgICB9O1xufVxuLyoqXG4gKiBTdGFydCBhIENvZGVCdWlsZCBCdWlsZCBhcyBhIHRhc2tcbiAqXG4gKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zdGVwLWZ1bmN0aW9ucy9sYXRlc3QvZGcvY29ubmVjdC1jb2RlYnVpbGQuaHRtbFxuICovXG5leHBvcnQgY2xhc3MgQ29kZUJ1aWxkU3RhcnRCdWlsZCBleHRlbmRzIHNmbi5UYXNrU3RhdGVCYXNlIHtcbiAgICBwcml2YXRlIHN0YXRpYyByZWFkb25seSBTVVBQT1JURURfSU5URUdSQVRJT05fUEFUVEVSTlM6IHNmbi5JbnRlZ3JhdGlvblBhdHRlcm5bXSA9IFtcbiAgICAgICAgc2ZuLkludGVncmF0aW9uUGF0dGVybi5SRVFVRVNUX1JFU1BPTlNFLFxuICAgICAgICBzZm4uSW50ZWdyYXRpb25QYXR0ZXJuLlJVTl9KT0IsXG4gICAgXTtcbiAgICBwcm90ZWN0ZWQgcmVhZG9ubHkgdGFza01ldHJpY3M/OiBzZm4uVGFza01ldHJpY3NDb25maWc7XG4gICAgcHJvdGVjdGVkIHJlYWRvbmx5IHRhc2tQb2xpY2llcz86IGlhbS5Qb2xpY3lTdGF0ZW1lbnRbXTtcbiAgICBwcml2YXRlIHJlYWRvbmx5IGludGVncmF0aW9uUGF0dGVybjogc2ZuLkludGVncmF0aW9uUGF0dGVybjtcbiAgICBjb25zdHJ1Y3RvcihzY29wZTogY2RrLkNvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJpdmF0ZSByZWFkb25seSBwcm9wczogQ29kZUJ1aWxkU3RhcnRCdWlsZFByb3BzKSB7XG4gICAgICAgIHN1cGVyKHNjb3BlLCBpZCwgcHJvcHMpO1xuICAgICAgICB0aGlzLmludGVncmF0aW9uUGF0dGVybiA9IHByb3BzLmludGVncmF0aW9uUGF0dGVybiA/PyBzZm4uSW50ZWdyYXRpb25QYXR0ZXJuLlJFUVVFU1RfUkVTUE9OU0U7XG4gICAgICAgIHZhbGlkYXRlUGF0dGVyblN1cHBvcnRlZCh0aGlzLmludGVncmF0aW9uUGF0dGVybiwgQ29kZUJ1aWxkU3RhcnRCdWlsZC5TVVBQT1JURURfSU5URUdSQVRJT05fUEFUVEVSTlMpO1xuICAgICAgICB0aGlzLnRhc2tNZXRyaWNzID0ge1xuICAgICAgICAgICAgbWV0cmljUHJlZml4U2luZ3VsYXI6ICdDb2RlQnVpbGRQcm9qZWN0JyxcbiAgICAgICAgICAgIG1ldHJpY1ByZWZpeFBsdXJhbDogJ0NvZGVCdWlsZFByb2plY3RzJyxcbiAgICAgICAgICAgIG1ldHJpY0RpbWVuc2lvbnM6IHtcbiAgICAgICAgICAgICAgICBQcm9qZWN0QXJuOiB0aGlzLnByb3BzLnByb2plY3QucHJvamVjdEFybixcbiAgICAgICAgICAgIH0sXG4gICAgICAgIH07XG4gICAgICAgIHRoaXMudGFza1BvbGljaWVzID0gdGhpcy5jb25maWd1cmVQb2xpY3lTdGF0ZW1lbnRzKCk7XG4gICAgfVxuICAgIHByaXZhdGUgY29uZmlndXJlUG9saWN5U3RhdGVtZW50cygpOiBpYW0uUG9saWN5U3RhdGVtZW50W10ge1xuICAgICAgICBsZXQgcG9saWN5U3RhdGVtZW50cyA9IFtcbiAgICAgICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgICAgICByZXNvdXJjZXM6IFt0aGlzLnByb3BzLnByb2plY3QucHJvamVjdEFybl0sXG4gICAgICAgICAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICAgICAgICAgICAnY29kZWJ1aWxkOlN0YXJ0QnVpbGQnLFxuICAgICAgICAgICAgICAgICAgICAnY29kZWJ1aWxkOlN0b3BCdWlsZCcsXG4gICAgICAgICAgICAgICAgICAgICdjb2RlYnVpbGQ6QmF0Y2hHZXRCdWlsZHMnLFxuICAgICAgICAgICAgICAgICAgICAnY29kZWJ1aWxkOkJhdGNoR2V0UmVwb3J0cycsXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH0pLFxuICAgICAgICBdO1xuICAgICAgICBpZiAodGhpcy5pbnRlZ3JhdGlvblBhdHRlcm4gPT09IHNmbi5JbnRlZ3JhdGlvblBhdHRlcm4uUlVOX0pPQikge1xuICAgICAgICAgICAgcG9saWN5U3RhdGVtZW50cy5wdXNoKG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgICAgICBhY3Rpb25zOiBbJ2V2ZW50czpQdXRUYXJnZXRzJywgJ2V2ZW50czpQdXRSdWxlJywgJ2V2ZW50czpEZXNjcmliZVJ1bGUnXSxcbiAgICAgICAgICAgICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgICAgICAgICAgICAgY2RrLlN0YWNrLm9mKHRoaXMpLmZvcm1hdEFybih7XG4gICAgICAgICAgICAgICAgICAgICAgICBzZXJ2aWNlOiAnZXZlbnRzJyxcbiAgICAgICAgICAgICAgICAgICAgICAgIHJlc291cmNlOiAncnVsZS9TdGVwRnVuY3Rpb25zR2V0RXZlbnRGb3JDb2RlQnVpbGRTdGFydEJ1aWxkUnVsZScsXG4gICAgICAgICAgICAgICAgICAgIH0pLFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9KSk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHBvbGljeVN0YXRlbWVudHM7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFByb3ZpZGVzIHRoZSBDb2RlQnVpbGQgU3RhcnRCdWlsZCBzZXJ2aWNlIGludGVncmF0aW9uIHRhc2sgY29uZmlndXJhdGlvblxuICAgICAqL1xuICAgIC8qKlxuICAgICAqIEBpbnRlcm5hbFxuICAgICAqL1xuICAgIHByb3RlY3RlZCBfcmVuZGVyVGFzaygpOiBhbnkge1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgUmVzb3VyY2U6IGludGVncmF0aW9uUmVzb3VyY2VBcm4oJ2NvZGVidWlsZCcsICdzdGFydEJ1aWxkJywgdGhpcy5pbnRlZ3JhdGlvblBhdHRlcm4pLFxuICAgICAgICAgICAgUGFyYW1ldGVyczogc2ZuLkZpZWxkVXRpbHMucmVuZGVyT2JqZWN0KHtcbiAgICAgICAgICAgICAgICBQcm9qZWN0TmFtZTogdGhpcy5wcm9wcy5wcm9qZWN0LnByb2plY3ROYW1lLFxuICAgICAgICAgICAgICAgIEVudmlyb25tZW50VmFyaWFibGVzT3ZlcnJpZGU6IHRoaXMucHJvcHMuZW52aXJvbm1lbnRWYXJpYWJsZXNPdmVycmlkZVxuICAgICAgICAgICAgICAgICAgICA/IHRoaXMuc2VyaWFsaXplRW52VmFyaWFibGVzKHRoaXMucHJvcHMuZW52aXJvbm1lbnRWYXJpYWJsZXNPdmVycmlkZSlcbiAgICAgICAgICAgICAgICAgICAgOiB1bmRlZmluZWQsXG4gICAgICAgICAgICB9KSxcbiAgICAgICAgfTtcbiAgICB9XG4gICAgcHJpdmF0ZSBzZXJpYWxpemVFbnZWYXJpYWJsZXMoZW52aXJvbm1lbnRWYXJpYWJsZXM6IHtcbiAgICAgICAgW25hbWU6IHN0cmluZ106IGNvZGVidWlsZC5CdWlsZEVudmlyb25tZW50VmFyaWFibGU7XG4gICAgfSkge1xuICAgICAgICByZXR1cm4gT2JqZWN0LmtleXMoZW52aXJvbm1lbnRWYXJpYWJsZXMpLm1hcChuYW1lID0+ICh7XG4gICAgICAgICAgICBOYW1lOiBuYW1lLFxuICAgICAgICAgICAgVHlwZTogZW52aXJvbm1lbnRWYXJpYWJsZXNbbmFtZV0udHlwZSB8fCBjb2RlYnVpbGQuQnVpbGRFbnZpcm9ubWVudFZhcmlhYmxlVHlwZS5QTEFJTlRFWFQsXG4gICAgICAgICAgICBWYWx1ZTogZW52aXJvbm1lbnRWYXJpYWJsZXNbbmFtZV0udmFsdWUsXG4gICAgICAgIH0pKTtcbiAgICB9XG59XG4iXX0=