"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaInvocationType = exports.LambdaInvoke = void 0;
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const task_utils_1 = require("../private/task-utils");
/**
 * Invoke a Lambda function as a Task
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-lambda.html
 */
class LambdaInvoke extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = (_a = props.integrationPattern) !== null && _a !== void 0 ? _a : sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, LambdaInvoke.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN
            && !sfn.FieldUtils.containsTaskToken(props.payload)) {
            throw new Error('Task Token is required in `payload` for callback. Use JsonPath.taskToken to set the token.');
        }
        if (props.payloadResponseOnly &&
            (props.integrationPattern || props.invocationType || props.clientContext || props.qualifier)) {
            throw new Error("The 'payloadResponseOnly' property cannot be used if 'integrationPattern', 'invocationType', 'clientContext', or 'qualifier' are specified.");
        }
        this.taskMetrics = {
            metricPrefixSingular: 'LambdaFunction',
            metricPrefixPlural: 'LambdaFunctions',
            metricDimensions: {
                LambdaFunctionArn: this.props.lambdaFunction.functionArn,
                ...(this.props.qualifier && { Qualifier: this.props.qualifier }),
            },
        };
        this.taskPolicies = [
            new iam.PolicyStatement({
                resources: [this.props.lambdaFunction.functionArn],
                actions: ['lambda:InvokeFunction'],
            }),
        ];
        if ((_b = props.retryOnServiceExceptions) !== null && _b !== void 0 ? _b : true) {
            // Best practice from https://docs.aws.amazon.com/step-functions/latest/dg/bp-lambda-serviceexception.html
            this.addRetry({
                errors: ['Lambda.ServiceException', 'Lambda.AWSLambdaException', 'Lambda.SdkClientException'],
                interval: cdk.Duration.seconds(2),
                maxAttempts: 6,
                backoffRate: 2,
            });
        }
    }
    /**
     * Provides the Lambda Invoke service integration task configuration
     */
    /**
     * @internal
     */
    _renderTask() {
        if (this.props.payloadResponseOnly) {
            return {
                Resource: this.props.lambdaFunction.functionArn,
                ...this.props.payload && { Parameters: sfn.FieldUtils.renderObject(this.props.payload.value) },
            };
        }
        else {
            return {
                Resource: task_utils_1.integrationResourceArn('lambda', 'invoke', this.integrationPattern),
                Parameters: sfn.FieldUtils.renderObject({
                    FunctionName: this.props.lambdaFunction.functionArn,
                    Payload: this.props.payload ? this.props.payload.value : sfn.TaskInput.fromDataAt('$').value,
                    InvocationType: this.props.invocationType,
                    ClientContext: this.props.clientContext,
                    Qualifier: this.props.qualifier,
                }),
            };
        }
    }
}
exports.LambdaInvoke = LambdaInvoke;
LambdaInvoke.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
/**
 * Invocation type of a Lambda
 */
var LambdaInvocationType;
(function (LambdaInvocationType) {
    /**
     * Invoke the function synchronously.
     *
     * Keep the connection open until the function returns a response or times out.
     * The API response includes the function response and additional data.
     */
    LambdaInvocationType["REQUEST_RESPONSE"] = "RequestResponse";
    /**
     * Invoke the function asynchronously.
     *
     * Send events that fail multiple times to the function's dead-letter queue (if it's configured).
     * The API response only includes a status code.
     */
    LambdaInvocationType["EVENT"] = "Event";
    /**
     * Validate parameter values and verify that the user or role has permission to invoke the function.
     */
    LambdaInvocationType["DRY_RUN"] = "DryRun";
})(LambdaInvocationType = exports.LambdaInvocationType || (exports.LambdaInvocationType = {}));
//# sourceMappingURL=data:application/json;base64,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