"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Chain = void 0;
const parallel_1 = require("./states/parallel");
/**
 * A collection of states to chain onto
 *
 * A Chain has a start and zero or more chainable ends. If there are
 * zero ends, calling next() on the Chain will fail.
 */
class Chain {
    constructor(startState, endStates, lastAdded) {
        this.lastAdded = lastAdded;
        this.id = lastAdded.id;
        this.startState = startState;
        this.endStates = endStates;
    }
    /**
     * Begin a new Chain from one chainable
     */
    static start(state) {
        return new Chain(state.startState, state.endStates, state);
    }
    /**
     * Make a Chain with the start from one chain and the ends from another
     */
    static sequence(start, next) {
        return new Chain(start.startState, next.endStates, next);
    }
    /**
     * Make a Chain with specific start and end states, and a last-added Chainable
     */
    static custom(startState, endStates, lastAdded) {
        return new Chain(startState, endStates, lastAdded);
    }
    /**
     * Continue normal execution with the given state
     */
    next(next) {
        if (this.endStates.length === 0) {
            throw new Error(`Cannot add to chain: last state in chain (${this.lastAdded.id}) does not allow it`);
        }
        for (const endState of this.endStates) {
            endState.next(next);
        }
        return new Chain(this.startState, next.endStates, next);
    }
    /**
     * Return a single state that encompasses all states in the chain
     *
     * This can be used to add error handling to a sequence of states.
     *
     * Be aware that this changes the result of the inner state machine
     * to be an array with the result of the state machine in it. Adjust
     * your paths accordingly. For example, change 'outputPath' to
     * '$[0]'.
     */
    toSingleState(id, props = {}) {
        return new parallel_1.Parallel(this.startState, id, props).branch(this);
    }
}
exports.Chain = Chain;
//# sourceMappingURL=data:application/json;base64,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