"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Condition = void 0;
/**
 * A Condition for use in a Choice state branch
 */
class Condition {
    /**
     * Matches if variable is present
     */
    static isPresent(variable) {
        return new VariableComparison(variable, ComparisonOperator.IsPresent, true);
    }
    /**
     * Matches if variable is not present
     */
    static isNotPresent(variable) {
        return new VariableComparison(variable, ComparisonOperator.IsPresent, false);
    }
    /**
     * Matches if variable is a string
     */
    static isString(variable) {
        return new VariableComparison(variable, ComparisonOperator.IsString, true);
    }
    /**
     * Matches if variable is not a string
     */
    static isNotString(variable) {
        return new VariableComparison(variable, ComparisonOperator.IsString, false);
    }
    /**
     * Matches if variable is numeric
     */
    static isNumeric(variable) {
        return new VariableComparison(variable, ComparisonOperator.IsNumeric, true);
    }
    /**
     * Matches if variable is not numeric
     */
    static isNotNumeric(variable) {
        return new VariableComparison(variable, ComparisonOperator.IsNumeric, false);
    }
    /**
     * Matches if variable is boolean
     */
    static isBoolean(variable) {
        return new VariableComparison(variable, ComparisonOperator.IsBoolean, true);
    }
    /**
     * Matches if variable is not boolean
     */
    static isNotBoolean(variable) {
        return new VariableComparison(variable, ComparisonOperator.IsBoolean, false);
    }
    /**
     * Matches if variable is a timestamp
     */
    static isTimestamp(variable) {
        return new VariableComparison(variable, ComparisonOperator.IsTimestamp, true);
    }
    /**
     * Matches if variable is not a timestamp
     */
    static isNotTimestamp(variable) {
        return new VariableComparison(variable, ComparisonOperator.IsTimestamp, false);
    }
    /**
     * Matches if variable is not null
     */
    static isNotNull(variable) {
        return new VariableComparison(variable, ComparisonOperator.IsNull, false);
    }
    /**
     * Matches if variable is Null
     */
    static isNull(variable) {
        return new VariableComparison(variable, ComparisonOperator.IsNull, true);
    }
    /**
     * Matches if a boolean field has the given value
     */
    static booleanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.BooleanEquals, value);
    }
    /**
     * Matches if a boolean field equals to a value at a given mapping path
     */
    static booleanEqualsJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.BooleanEqualsPath, value);
    }
    /**
     * Matches if a string field equals to a value at a given mapping path
     */
    static stringEqualsJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringEqualsPath, value);
    }
    /**
     * Matches if a string field has the given value
     */
    static stringEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringEquals, value);
    }
    /**
     * Matches if a string field sorts before a given value
     */
    static stringLessThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringLessThan, value);
    }
    /**
     * Matches if a string field sorts before a given value at a particular mapping
     */
    static stringLessThanJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringLessThanPath, value);
    }
    /**
     * Matches if a string field sorts equal to or before a given value
     */
    static stringLessThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringLessThanEquals, value);
    }
    /**
     * Matches if a string field sorts equal to or before a given mapping
     */
    static stringLessThanEqualsJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringLessThanEqualsPath, value);
    }
    /**
     * Matches if a string field sorts after a given value
     */
    static stringGreaterThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringGreaterThan, value);
    }
    /**
     * Matches if a string field sorts after a value at a given mapping path
     */
    static stringGreaterThanJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringGreaterThanPath, value);
    }
    /**
     * Matches if a string field sorts after or equal to value at a given mapping path
     */
    static stringGreaterThanEqualsJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringGreaterThanEqualsPath, value);
    }
    /**
     * Matches if a string field sorts after or equal to a given value
     */
    static stringGreaterThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringGreaterThanEquals, value);
    }
    /**
     * Matches if a numeric field has the given value
     */
    static numberEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericEquals, value);
    }
    /**
     * Matches if a numeric field has the value in a given mapping path
     */
    static numberEqualsJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericEqualsPath, value);
    }
    /**
     * Matches if a numeric field is less than the given value
     */
    static numberLessThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericLessThan, value);
    }
    /**
     * Matches if a numeric field is less than the value at the given mapping path
     */
    static numberLessThanJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericLessThanPath, value);
    }
    /**
     * Matches if a numeric field is less than or equal to the given value
     */
    static numberLessThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericLessThanEquals, value);
    }
    /**
     * Matches if a numeric field is less than or equal to the numeric value at given mapping path
     */
    static numberLessThanEqualsJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericLessThanEqualsPath, value);
    }
    /**
     * Matches if a numeric field is greater than the given value
     */
    static numberGreaterThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericGreaterThan, value);
    }
    /**
     * Matches if a numeric field is greater than the value at a given mapping path
     */
    static numberGreaterThanJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericGreaterThanPath, value);
    }
    /**
     * Matches if a numeric field is greater than or equal to the given value
     */
    static numberGreaterThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericGreaterThanEquals, value);
    }
    /**
     * Matches if a numeric field is greater than or equal to the value at a given mapping path
     */
    static numberGreaterThanEqualsJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericGreaterThanEqualsPath, value);
    }
    /**
     * Matches if a timestamp field is the same time as the given timestamp
     */
    static timestampEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampEquals, value);
    }
    /**
     * Matches if a timestamp field is the same time as the timestamp at a given mapping path
     */
    static timestampEqualsJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampEqualsPath, value);
    }
    /**
     * Matches if a timestamp field is before the given timestamp
     */
    static timestampLessThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampLessThan, value);
    }
    /**
     * Matches if a timestamp field is before the timestamp at a given mapping path
     */
    static timestampLessThanJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampLessThanPath, value);
    }
    /**
     * Matches if a timestamp field is before or equal to the given timestamp
     */
    static timestampLessThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampLessThanEquals, value);
    }
    /**
     * Matches if a timestamp field is before or equal to the timestamp at a given mapping path
     */
    static timestampLessThanEqualsJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampLessThanEqualsPath, value);
    }
    /**
     * Matches if a timestamp field is after the given timestamp
     */
    static timestampGreaterThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampGreaterThan, value);
    }
    /**
     * Matches if a timestamp field is after the timestamp at a given mapping path
     */
    static timestampGreaterThanJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampGreaterThanPath, value);
    }
    /**
     * Matches if a timestamp field is after or equal to the given timestamp
     */
    static timestampGreaterThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampGreaterThanEquals, value);
    }
    /**
     * Matches if a timestamp field is after or equal to the timestamp at a given mapping path
     */
    static timestampGreaterThanEqualsJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampGreaterThanEqualsPath, value);
    }
    /**
     * Matches if a field matches a string pattern that can contain a wild card (*) e.g: log-*.txt or *LATEST*.
     * No other characters other than "*" have any special meaning - * can be escaped: \\*
     */
    static stringMatches(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringMatches, value);
    }
    /**
     * Combine two or more conditions with a logical AND
     */
    static and(...conditions) {
        return new CompoundCondition(CompoundOperator.And, ...conditions);
    }
    /**
     * Combine two or more conditions with a logical OR
     */
    static or(...conditions) {
        return new CompoundCondition(CompoundOperator.Or, ...conditions);
    }
    /**
     * Negate a condition
     */
    static not(condition) {
        return new NotCondition(condition);
    }
}
exports.Condition = Condition;
/**
 * Comparison Operator types
 */
var ComparisonOperator;
(function (ComparisonOperator) {
    ComparisonOperator[ComparisonOperator["StringEquals"] = 0] = "StringEquals";
    ComparisonOperator[ComparisonOperator["StringEqualsPath"] = 1] = "StringEqualsPath";
    ComparisonOperator[ComparisonOperator["StringLessThan"] = 2] = "StringLessThan";
    ComparisonOperator[ComparisonOperator["StringLessThanPath"] = 3] = "StringLessThanPath";
    ComparisonOperator[ComparisonOperator["StringGreaterThan"] = 4] = "StringGreaterThan";
    ComparisonOperator[ComparisonOperator["StringGreaterThanPath"] = 5] = "StringGreaterThanPath";
    ComparisonOperator[ComparisonOperator["StringLessThanEquals"] = 6] = "StringLessThanEquals";
    ComparisonOperator[ComparisonOperator["StringLessThanEqualsPath"] = 7] = "StringLessThanEqualsPath";
    ComparisonOperator[ComparisonOperator["StringGreaterThanEquals"] = 8] = "StringGreaterThanEquals";
    ComparisonOperator[ComparisonOperator["StringGreaterThanEqualsPath"] = 9] = "StringGreaterThanEqualsPath";
    ComparisonOperator[ComparisonOperator["NumericEquals"] = 10] = "NumericEquals";
    ComparisonOperator[ComparisonOperator["NumericEqualsPath"] = 11] = "NumericEqualsPath";
    ComparisonOperator[ComparisonOperator["NumericLessThan"] = 12] = "NumericLessThan";
    ComparisonOperator[ComparisonOperator["NumericLessThanPath"] = 13] = "NumericLessThanPath";
    ComparisonOperator[ComparisonOperator["NumericGreaterThan"] = 14] = "NumericGreaterThan";
    ComparisonOperator[ComparisonOperator["NumericGreaterThanPath"] = 15] = "NumericGreaterThanPath";
    ComparisonOperator[ComparisonOperator["NumericLessThanEquals"] = 16] = "NumericLessThanEquals";
    ComparisonOperator[ComparisonOperator["NumericLessThanEqualsPath"] = 17] = "NumericLessThanEqualsPath";
    ComparisonOperator[ComparisonOperator["NumericGreaterThanEquals"] = 18] = "NumericGreaterThanEquals";
    ComparisonOperator[ComparisonOperator["NumericGreaterThanEqualsPath"] = 19] = "NumericGreaterThanEqualsPath";
    ComparisonOperator[ComparisonOperator["BooleanEquals"] = 20] = "BooleanEquals";
    ComparisonOperator[ComparisonOperator["BooleanEqualsPath"] = 21] = "BooleanEqualsPath";
    ComparisonOperator[ComparisonOperator["TimestampEquals"] = 22] = "TimestampEquals";
    ComparisonOperator[ComparisonOperator["TimestampEqualsPath"] = 23] = "TimestampEqualsPath";
    ComparisonOperator[ComparisonOperator["TimestampLessThan"] = 24] = "TimestampLessThan";
    ComparisonOperator[ComparisonOperator["TimestampLessThanPath"] = 25] = "TimestampLessThanPath";
    ComparisonOperator[ComparisonOperator["TimestampGreaterThan"] = 26] = "TimestampGreaterThan";
    ComparisonOperator[ComparisonOperator["TimestampGreaterThanPath"] = 27] = "TimestampGreaterThanPath";
    ComparisonOperator[ComparisonOperator["TimestampLessThanEquals"] = 28] = "TimestampLessThanEquals";
    ComparisonOperator[ComparisonOperator["TimestampLessThanEqualsPath"] = 29] = "TimestampLessThanEqualsPath";
    ComparisonOperator[ComparisonOperator["TimestampGreaterThanEquals"] = 30] = "TimestampGreaterThanEquals";
    ComparisonOperator[ComparisonOperator["TimestampGreaterThanEqualsPath"] = 31] = "TimestampGreaterThanEqualsPath";
    ComparisonOperator[ComparisonOperator["IsNull"] = 32] = "IsNull";
    ComparisonOperator[ComparisonOperator["IsBoolean"] = 33] = "IsBoolean";
    ComparisonOperator[ComparisonOperator["IsNumeric"] = 34] = "IsNumeric";
    ComparisonOperator[ComparisonOperator["IsString"] = 35] = "IsString";
    ComparisonOperator[ComparisonOperator["IsTimestamp"] = 36] = "IsTimestamp";
    ComparisonOperator[ComparisonOperator["IsPresent"] = 37] = "IsPresent";
    ComparisonOperator[ComparisonOperator["StringMatches"] = 38] = "StringMatches";
})(ComparisonOperator || (ComparisonOperator = {}));
/**
 * Compound Operator types
 */
var CompoundOperator;
(function (CompoundOperator) {
    CompoundOperator[CompoundOperator["And"] = 0] = "And";
    CompoundOperator[CompoundOperator["Or"] = 1] = "Or";
})(CompoundOperator || (CompoundOperator = {}));
/**
 * Scalar comparison
 */
class VariableComparison extends Condition {
    constructor(variable, comparisonOperator, value) {
        super();
        this.variable = variable;
        this.comparisonOperator = comparisonOperator;
        this.value = value;
        if (!/^\$|(\$[.[])/.test(variable)) {
            throw new Error(`Variable reference must be '$', start with '$.', or start with '$[', got '${variable}'`);
        }
    }
    renderCondition() {
        return {
            Variable: this.variable,
            [ComparisonOperator[this.comparisonOperator]]: this.value,
        };
    }
}
/**
 * Logical compound condition
 */
class CompoundCondition extends Condition {
    constructor(operator, ...conditions) {
        super();
        this.operator = operator;
        this.conditions = conditions;
        if (conditions.length === 0) {
            throw new Error('Must supply at least one inner condition for a logical combination');
        }
    }
    renderCondition() {
        return {
            [CompoundOperator[this.operator]]: this.conditions.map(c => c.renderCondition()),
        };
    }
}
/**
 * Logical unary condition
 */
class NotCondition extends Condition {
    constructor(comparisonOperation) {
        super();
        this.comparisonOperation = comparisonOperation;
    }
    renderCondition() {
        return {
            Not: this.comparisonOperation.renderCondition(),
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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