"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.jsonPathString = exports.recurseObject = exports.findReferencedPaths = exports.renderObject = exports.JsonPathToken = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const JSON_PATH_TOKEN_SYMBOL = Symbol.for('@aws-cdk/aws-stepfunctions.JsonPathToken');
class JsonPathToken {
    constructor(path) {
        this.path = path;
        this.creationStack = core_1.captureStackTrace();
        this.displayHint = path.replace(/^[^a-zA-Z]+/, '');
        Object.defineProperty(this, JSON_PATH_TOKEN_SYMBOL, { value: true });
    }
    static isJsonPathToken(x) {
        return x[JSON_PATH_TOKEN_SYMBOL] === true;
    }
    resolve(_ctx) {
        return this.path;
    }
    toString() {
        return core_1.Token.asString(this, { displayHint: this.displayHint });
    }
    toJSON() {
        return `<path:${this.path}>`;
    }
}
exports.JsonPathToken = JsonPathToken;
/**
 * Deep render a JSON object to expand JSON path fields, updating the key to end in '.$'
 */
function renderObject(obj) {
    return recurseObject(obj, {
        handleString: renderString,
        handleList: renderStringList,
        handleNumber: renderNumber,
        handleBoolean: renderBoolean,
    });
}
exports.renderObject = renderObject;
/**
 * Return all JSON paths that are used in the given structure
 */
function findReferencedPaths(obj) {
    const found = new Set();
    recurseObject(obj, {
        handleString(_key, x) {
            const path = jsonPathString(x);
            if (path !== undefined) {
                found.add(path);
            }
            return {};
        },
        handleList(_key, x) {
            const path = jsonPathStringList(x);
            if (path !== undefined) {
                found.add(path);
            }
            return {};
        },
        handleNumber(_key, x) {
            const path = jsonPathNumber(x);
            if (path !== undefined) {
                found.add(path);
            }
            return {};
        },
        handleBoolean(_key, _x) {
            return {};
        },
    });
    return found;
}
exports.findReferencedPaths = findReferencedPaths;
function recurseObject(obj, handlers) {
    if (obj === undefined) {
        return undefined;
    }
    const ret = {};
    for (const [key, value] of Object.entries(obj)) {
        if (typeof value === 'string') {
            Object.assign(ret, handlers.handleString(key, value));
        }
        else if (typeof value === 'number') {
            Object.assign(ret, handlers.handleNumber(key, value));
        }
        else if (Array.isArray(value)) {
            Object.assign(ret, recurseArray(key, value, handlers));
        }
        else if (typeof value === 'boolean') {
            Object.assign(ret, handlers.handleBoolean(key, value));
        }
        else if (value === null || value === undefined) {
            // Nothing
        }
        else if (typeof value === 'object') {
            ret[key] = recurseObject(value, handlers);
        }
    }
    return ret;
}
exports.recurseObject = recurseObject;
/**
 * Render an array that may or may not contain a string list token
 */
function recurseArray(key, arr, handlers) {
    if (isStringArray(arr)) {
        const path = jsonPathStringList(arr);
        if (path !== undefined) {
            return handlers.handleList(key, arr);
        }
        // Fall through to correctly reject encoded strings inside an array.
        // They cannot be represented because there is no key to append a '.$' to.
    }
    return {
        [key]: arr.map(value => {
            if ((typeof value === 'string' && jsonPathString(value) !== undefined)
                || (typeof value === 'number' && jsonPathNumber(value) !== undefined)
                || (isStringArray(value) && jsonPathStringList(value) !== undefined)) {
                throw new Error('Cannot use JsonPath fields in an array, they must be used in objects');
            }
            if (typeof value === 'object' && value !== null) {
                return recurseObject(value, handlers);
            }
            return value;
        }),
    };
}
function isStringArray(x) {
    return Array.isArray(x) && x.every(el => typeof el === 'string');
}
/**
 * Render a parameter string
 *
 * If the string value starts with '$.', render it as a path string, otherwise as a direct string.
 */
function renderString(key, value) {
    const path = jsonPathString(value);
    if (path !== undefined) {
        return { [key + '.$']: path };
    }
    else {
        return { [key]: value };
    }
}
/**
 * Render a parameter string list
 *
 * If the string value starts with '$.', render it as a path string, otherwise as a direct string.
 */
function renderStringList(key, value) {
    const path = jsonPathStringList(value);
    if (path !== undefined) {
        return { [key + '.$']: path };
    }
    else {
        return { [key]: value };
    }
}
/**
 * Render a parameter number
 *
 * If the string value starts with '$.', render it as a path string, otherwise as a direct string.
 */
function renderNumber(key, value) {
    const path = jsonPathNumber(value);
    if (path !== undefined) {
        return { [key + '.$']: path };
    }
    else {
        return { [key]: value };
    }
}
/**
 * Render a parameter boolean
 */
function renderBoolean(key, value) {
    return { [key]: value };
}
/**
 * If the indicated string is an encoded JSON path, return the path
 *
 * Otherwise return undefined.
 */
function jsonPathString(x) {
    const fragments = core_1.Tokenization.reverseString(x);
    const jsonPathTokens = fragments.tokens.filter(JsonPathToken.isJsonPathToken);
    if (jsonPathTokens.length > 0 && fragments.length > 1) {
        throw new Error(`Field references must be the entire string, cannot concatenate them (found '${x}')`);
    }
    if (jsonPathTokens.length > 0) {
        return jsonPathTokens[0].path;
    }
    return undefined;
}
exports.jsonPathString = jsonPathString;
/**
 * If the indicated string list is an encoded JSON path, return the path
 *
 * Otherwise return undefined.
 */
function jsonPathStringList(x) {
    return pathFromToken(core_1.Tokenization.reverseList(x));
}
/**
 * If the indicated number is an encoded JSON path, return the path
 *
 * Otherwise return undefined.
 */
function jsonPathNumber(x) {
    return pathFromToken(core_1.Tokenization.reverseNumber(x));
}
function pathFromToken(token) {
    return token && (JsonPathToken.isJsonPathToken(token) ? token.path : undefined);
}
//# sourceMappingURL=data:application/json;base64,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