"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Choice = void 0;
const chain_1 = require("../chain");
const state_type_1 = require("./private/state-type");
const state_1 = require("./state");
/**
 * Define a Choice in the state machine
 *
 * A choice state can be used to make decisions based on the execution
 * state.
 */
class Choice extends state_1.State {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.endStates = [];
    }
    /**
     * If the given condition matches, continue execution with the given state
     */
    when(condition, next) {
        super.addChoice(condition, next.startState);
        return this;
    }
    /**
     * If none of the given conditions match, continue execution with the given state
     *
     * If no conditions match and no otherwise() has been given, an execution
     * error will be raised.
     */
    otherwise(def) {
        super.makeDefault(def.startState);
        return this;
    }
    /**
     * Return a Chain that contains all reachable end states from this Choice
     *
     * Use this to combine all possible choice paths back.
     */
    afterwards(options = {}) {
        const endStates = state_1.State.filterNextables(state_1.State.findReachableEndStates(this, { includeErrorHandlers: options.includeErrorHandlers }));
        if (options.includeOtherwise && this.defaultChoice) {
            throw new Error(`'includeOtherwise' set but Choice state ${this.stateId} already has an 'otherwise' transition`);
        }
        if (options.includeOtherwise) {
            endStates.push(new DefaultAsNext(this));
        }
        return chain_1.Chain.custom(this, endStates, this);
    }
    /**
     * Return the Amazon States Language object for this state
     */
    toStateJson() {
        return {
            Type: state_type_1.StateType.CHOICE,
            Comment: this.comment,
            ...this.renderInputOutput(),
            ...this.renderChoices(),
        };
    }
}
exports.Choice = Choice;
/**
 * Adapter to make the .otherwise() transition settable through .next()
 */
class DefaultAsNext {
    constructor(choice) {
        this.choice = choice;
    }
    next(state) {
        this.choice.otherwise(state);
        return chain_1.Chain.sequence(this.choice, state);
    }
}
//# sourceMappingURL=data:application/json;base64,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