"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Canary = exports.Runtime = exports.Test = void 0;
const crypto = require("crypto");
const aws_cloudwatch_1 = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const synthetics_generated_1 = require("./synthetics.generated");
/**
 * Specify a test that the canary should run
 */
class Test {
    /**
     * Construct a Test property
     *
     * @param code The code that the canary should run
     * @param handler The handler of the canary
     */
    constructor(code, handler) {
        this.code = code;
        this.handler = handler;
    }
    /**
     * Specify a custom test with your own code
     *
     * @returns `Test` associated with the specified Code object
     * @param options The configuration options
     */
    static custom(options) {
        Test.validateHandler(options.handler);
        return new Test(options.code, options.handler);
    }
    /**
     * Verifies that the given handler ends in '.handler'. Returns the handler if successful and
     * throws an error if not.
     *
     * @param handler - the handler given by the user
     */
    static validateHandler(handler) {
        if (!handler.endsWith('.handler')) {
            throw new Error(`Canary Handler must end in '.handler' (${handler})`);
        }
        if (handler.length > 21) {
            throw new Error(`Canary Handler must be less than 21 characters (${handler})`);
        }
    }
}
exports.Test = Test;
/**
 * Runtime options for a canary
 */
class Runtime {
    /**
     * @param name The name of the runtime version
     */
    constructor(name) {
        this.name = name;
    }
}
exports.Runtime = Runtime;
/**
 * `syn-1.0` includes the following:
 *
 * - Synthetics library 1.0
 * - Synthetics handler code 1.0
 * - Lambda runtime Node.js 10.x
 * - Puppeteer-core version 1.14.0
 * - The Chromium version that matches Puppeteer-core 1.14.0
 *
 * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_Library.html#CloudWatch_Synthetics_runtimeversion-1.0
 */
Runtime.SYNTHETICS_1_0 = new Runtime('syn-1.0');
/**
 * `syn-nodejs-2.0` includes the following:
 * - Lambda runtime Node.js 10.x
 * - Puppeteer-core version 3.3.0
 * - Chromium version 81.0.4044.0
 *
 * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_Library.html#CloudWatch_Synthetics_runtimeversion-2.0
 */
Runtime.SYNTHETICS_NODEJS_2_0 = new Runtime('syn-nodejs-2.0');
/**
 * Define a new Canary
 */
class Canary extends cdk.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
        if (props.canaryName && !cdk.Token.isUnresolved(props.canaryName)) {
            validateName(props.canaryName);
        }
        super(scope, id, {
            physicalName: props.canaryName || cdk.Lazy.stringValue({
                produce: () => this.generateUniqueName(),
            }),
        });
        this.artifactsBucket = (_b = (_a = props.artifactsBucketLocation) === null || _a === void 0 ? void 0 : _a.bucket) !== null && _b !== void 0 ? _b : new s3.Bucket(this, 'ArtifactsBucket', {
            encryption: s3.BucketEncryption.KMS_MANAGED,
        });
        this.role = (_c = props.role) !== null && _c !== void 0 ? _c : this.createDefaultRole((_d = props.artifactsBucketLocation) === null || _d === void 0 ? void 0 : _d.prefix);
        const resource = new synthetics_generated_1.CfnCanary(this, 'Resource', {
            artifactS3Location: this.artifactsBucket.s3UrlForObject((_e = props.artifactsBucketLocation) === null || _e === void 0 ? void 0 : _e.prefix),
            executionRoleArn: this.role.roleArn,
            startCanaryAfterCreation: (_f = props.startAfterCreation) !== null && _f !== void 0 ? _f : true,
            runtimeVersion: (_h = (_g = props.runtime) === null || _g === void 0 ? void 0 : _g.name) !== null && _h !== void 0 ? _h : Runtime.SYNTHETICS_NODEJS_2_0.name,
            name: this.physicalName,
            schedule: this.createSchedule(props),
            failureRetentionPeriod: (_j = props.failureRetentionPeriod) === null || _j === void 0 ? void 0 : _j.toDays(),
            successRetentionPeriod: (_k = props.successRetentionPeriod) === null || _k === void 0 ? void 0 : _k.toDays(),
            code: this.createCode(props),
        });
        this.canaryId = resource.attrId;
        this.canaryState = resource.attrState;
        this.canaryName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * Measure the Duration of a single canary run, in seconds.
     *
     * @param options - configuration options for the metric
     *
     * @default avg over 5 minutes
     */
    metricDuration(options) {
        return this.metric('Duration', options);
    }
    /**
     * Measure the percentage of successful canary runs.
     *
     * @param options - configuration options for the metric
     *
     * @default avg over 5 minutes
     */
    metricSuccessPercent(options) {
        return this.metric('SuccessPercent', options);
    }
    /**
     * Measure the number of failed canary runs over a given time period.
     *
     * @param options - configuration options for the metric
     *
     * @default avg over 5 minutes
     */
    metricFailed(options) {
        return this.metric('Failed', options);
    }
    /**
     * @param metricName - the name of the metric
     * @param options - configuration options for the metric
     *
     * @returns a CloudWatch metric associated with the canary.
     * @default avg over 5 minutes
     */
    metric(metricName, options) {
        return new aws_cloudwatch_1.Metric({
            metricName,
            namespace: 'CloudWatchSynthetics',
            dimensions: { CanaryName: this.canaryName },
            statistic: 'avg',
            ...options,
        }).attachTo(this);
    }
    /**
     * Returns a default role for the canary
     */
    createDefaultRole(prefix) {
        // Created role will need these policies to run the Canary.
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-synthetics-canary.html#cfn-synthetics-canary-executionrolearn
        const policy = new iam.PolicyDocument({
            statements: [
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: ['s3:ListAllMyBuckets'],
                }),
                new iam.PolicyStatement({
                    resources: [this.artifactsBucket.arnForObjects(`${prefix ? prefix + '/*' : '*'}`)],
                    actions: ['s3:PutObject', 's3:GetBucketLocation'],
                }),
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: ['cloudwatch:PutMetricData'],
                    conditions: { StringEquals: { 'cloudwatch:namespace': 'CloudWatchSynthetics' } },
                }),
                new iam.PolicyStatement({
                    resources: ['arn:aws:logs:::*'],
                    actions: ['logs:CreateLogStream', 'logs:CreateLogGroup', 'logs:PutLogEvents'],
                }),
            ],
        });
        return new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                canaryPolicy: policy,
            },
        });
    }
    /**
     * Returns the code object taken in by the canary resource.
     */
    createCode(props) {
        var _a, _b, _c;
        const codeConfig = {
            handler: props.test.handler,
            ...props.test.code.bind(this, props.test.handler),
        };
        return {
            handler: codeConfig.handler,
            script: codeConfig.inlineCode,
            s3Bucket: (_a = codeConfig.s3Location) === null || _a === void 0 ? void 0 : _a.bucketName,
            s3Key: (_b = codeConfig.s3Location) === null || _b === void 0 ? void 0 : _b.objectKey,
            s3ObjectVersion: (_c = codeConfig.s3Location) === null || _c === void 0 ? void 0 : _c.objectVersion,
        };
    }
    /**
     * Returns a canary schedule object
     */
    createSchedule(props) {
        var _a, _b, _c, _d;
        return {
            durationInSeconds: String(`${(_b = (_a = props.timeToLive) === null || _a === void 0 ? void 0 : _a.toSeconds()) !== null && _b !== void 0 ? _b : 0}`),
            expression: (_d = (_c = props.schedule) === null || _c === void 0 ? void 0 : _c.expressionString) !== null && _d !== void 0 ? _d : 'rate(5 minutes)',
        };
    }
    /**
     * Creates a unique name for the canary. The generated name is the physical ID of the canary.
     */
    generateUniqueName() {
        const name = this.node.uniqueId.toLowerCase().replace(' ', '-');
        if (name.length <= 21) {
            return name;
        }
        else {
            return name.substring(0, 15) + nameHash(name);
        }
    }
}
exports.Canary = Canary;
/**
 * Take a hash of the given name.
 *
 * @param name the name to be hashed
 */
function nameHash(name) {
    const md5 = crypto.createHash('sha256').update(name).digest('hex');
    return md5.slice(0, 6);
}
const nameRegex = /^[0-9a-z_\-]+$/;
/**
 * Verifies that the name fits the regex expression: ^[0-9a-z_\-]+$.
 *
 * @param name - the given name of the canary
 */
function validateName(name) {
    if (name.length > 21) {
        throw new Error(`Canary name is too large, must be between 1 and 21 characters, but is ${name.length} (got "${name}")`);
    }
    if (!nameRegex.test(name)) {
        throw new Error(`Canary name must be lowercase, numbers, hyphens, or underscores (got "${name}")`);
    }
}
//# sourceMappingURL=data:application/json;base64,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