"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3Code = exports.InlineCode = exports.AssetCode = exports.Code = void 0;
const fs = require("fs");
const path = require("path");
const s3_assets = require("../../aws-s3-assets"); // Automatically re-written from '@aws-cdk/aws-s3-assets'
/**
 * The code the canary should execute
 */
class Code {
    /**
     * Specify code inline.
     *
     * @param code The actual handler code (limited to 4KiB)
     *
     * @returns `InlineCode` with inline code.
     */
    static fromInline(code) {
        return new InlineCode(code);
    }
    /**
     * Specify code from a local path. Path must include the folder structure `nodejs/node_modules/myCanaryFilename.js`.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html#CloudWatch_Synthetics_Canaries_write_from_scratch
     *
     * @param assetPath Either a directory or a .zip file
     *
     * @returns `AssetCode` associated with the specified path.
     */
    static fromAsset(assetPath, options) {
        return new AssetCode(assetPath, options);
    }
    /**
     * Specify code from an s3 bucket. The object in the s3 bucket must be a .zip file that contains
     * the structure `nodejs/node_modules/myCanaryFilename.js`.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html#CloudWatch_Synthetics_Canaries_write_from_scratch
     *
     * @param bucket The S3 bucket
     * @param key The object key
     * @param objectVersion Optional S3 object version
     *
     * @returns `S3Code` associated with the specified S3 object.
     */
    static fromBucket(bucket, key, objectVersion) {
        return new S3Code(bucket, key, objectVersion);
    }
}
exports.Code = Code;
/**
 * Canary code from an Asset
 */
class AssetCode extends Code {
    /**
     * @param assetPath The path to the asset file or directory.
     */
    constructor(assetPath, options) {
        super();
        this.assetPath = assetPath;
        this.options = options;
        if (!fs.existsSync(this.assetPath)) {
            throw new Error(`${this.assetPath} is not a valid path`);
        }
    }
    bind(scope, handler) {
        this.validateCanaryAsset(handler);
        // If the same AssetCode is used multiple times, retain only the first instantiation.
        if (!this.asset) {
            this.asset = new s3_assets.Asset(scope, 'Code', {
                path: this.assetPath,
                ...this.options,
            });
        }
        return {
            s3Location: {
                bucketName: this.asset.s3BucketName,
                objectKey: this.asset.s3ObjectKey,
            },
        };
    }
    /**
     * Validates requirements specified by the canary resource. For example, the canary code with handler `index.handler`
     * must be found in the file structure `nodejs/node_modules/index.js`.
     *
     * Requires path to be either zip file or directory.
     * Requires asset directory to have the structure 'nodejs/node_modules'.
     * Requires canary file to be directly inside node_modules folder.
     * Requires canary file name matches the handler name.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html
     *
     * @param handler the canary handler
     */
    validateCanaryAsset(handler) {
        if (path.extname(this.assetPath) !== '.zip') {
            if (!fs.lstatSync(this.assetPath).isDirectory()) {
                throw new Error(`Asset must be a .zip file or a directory (${this.assetPath})`);
            }
            const filename = `${handler.split('.')[0]}.js`;
            if (!fs.existsSync(path.join(this.assetPath, 'nodejs', 'node_modules', filename))) {
                throw new Error(`The canary resource requires that the handler is present at "nodejs/node_modules/${filename}" but not found at ${this.assetPath} (https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html#CloudWatch_Synthetics_Canaries_write_from_scratch)`);
            }
        }
    }
}
exports.AssetCode = AssetCode;
/**
 * Canary code from an inline string.
 */
class InlineCode extends Code {
    constructor(code) {
        super();
        this.code = code;
        if (code.length === 0) {
            throw new Error('Canary inline code cannot be empty');
        }
    }
    bind(_scope, handler) {
        if (handler !== 'index.handler') {
            throw new Error(`The handler for inline code must be "index.handler" (got "${handler}")`);
        }
        return {
            inlineCode: this.code,
        };
    }
}
exports.InlineCode = InlineCode;
/**
 * S3 bucket path to the code zip file
 */
class S3Code extends Code {
    constructor(bucket, key, objectVersion) {
        super();
        this.bucket = bucket;
        this.key = key;
        this.objectVersion = objectVersion;
    }
    bind(_scope, _handler) {
        return {
            s3Location: {
                bucketName: this.bucket.bucketName,
                objectKey: this.key,
                objectVersion: this.objectVersion,
            },
        };
    }
}
exports.S3Code = S3Code;
//# sourceMappingURL=data:application/json;base64,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