"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Manifest = exports.VERSION_MISMATCH = void 0;
const fs = require("fs");
const jsonschema = require("jsonschema");
const semver = require("semver");
const assembly = require("./cloud-assembly");
/* eslint-disable @typescript-eslint/no-var-requires */
/* eslint-disable @typescript-eslint/no-require-imports */
// this prefix is used by the CLI to identify this specific error.
// in which case we want to instruct the user to upgrade his CLI.
// see exec.ts#createAssembly
exports.VERSION_MISMATCH = 'Cloud assembly schema version mismatch';
const ASSETS_SCHEMA = require('../schema/assets.schema.json');
const ASSEMBLY_SCHEMA = require('../schema/cloud-assembly.schema.json');
/**
 * Version is shared for both manifests
 */
const SCHEMA_VERSION = require('../schema/cloud-assembly.version.json').version;
/**
 * Protocol utility class.
 */
class Manifest {
    /**
     * Validates and saves the cloud assembly manifest to file.
     *
     * @param manifest - manifest.
     * @param filePath - output file path.
     */
    static saveAssemblyManifest(manifest, filePath) {
        Manifest.saveManifest(manifest, filePath, ASSEMBLY_SCHEMA, Manifest.patchStackTagsOnWrite);
    }
    /**
     * Load and validates the cloud assembly manifest from file.
     *
     * @param filePath - path to the manifest file.
     */
    static loadAssemblyManifest(filePath) {
        return Manifest.loadManifest(filePath, ASSEMBLY_SCHEMA, Manifest.patchStackTagsOnRead);
    }
    /**
     * Validates and saves the asset manifest to file.
     *
     * @param manifest - manifest.
     * @param filePath - output file path.
     */
    static saveAssetManifest(manifest, filePath) {
        Manifest.saveManifest(manifest, filePath, ASSETS_SCHEMA, Manifest.patchStackTagsOnRead);
    }
    /**
     * Load and validates the asset manifest from file.
     *
     * @param filePath - path to the manifest file.
     */
    static loadAssetManifest(filePath) {
        return this.loadManifest(filePath, ASSETS_SCHEMA);
    }
    /**
     * Fetch the current schema version number.
     */
    static version() {
        return SCHEMA_VERSION;
    }
    /**
     * Deprecated
     * @deprecated use `saveAssemblyManifest()`
     */
    static save(manifest, filePath) { return this.saveAssemblyManifest(manifest, filePath); }
    /**
     * Deprecated
     * @deprecated use `loadAssemblyManifest()`
     */
    static load(filePath) { return this.loadAssemblyManifest(filePath); }
    static validate(manifest, schema) {
        function parseVersion(version) {
            const ver = semver.valid(version);
            if (!ver) {
                throw new Error(`Invalid semver string: "${version}"`);
            }
            return ver;
        }
        const maxSupported = parseVersion(Manifest.version());
        const actual = parseVersion(manifest.version);
        // first validate the version should be accepted.
        if (semver.gt(actual, maxSupported)) {
            // we use a well known error prefix so that the CLI can identify this specific error
            // and print some more context to the user.
            throw new Error(`${exports.VERSION_MISMATCH}: Maximum schema version supported is ${maxSupported}, but found ${actual}`);
        }
        // now validate the format is good.
        const validator = new jsonschema.Validator();
        const result = validator.validate(manifest, schema, {
            // does exist but is not in the TypeScript definitions
            nestedErrors: true,
            allowUnknownAttributes: false,
        });
        if (!result.valid) {
            throw new Error(`Invalid assembly manifest:\n${result}`);
        }
    }
    static saveManifest(manifest, filePath, schema, preprocess) {
        let withVersion = { ...manifest, version: Manifest.version() };
        Manifest.validate(withVersion, schema);
        if (preprocess) {
            withVersion = preprocess(withVersion);
        }
        fs.writeFileSync(filePath, JSON.stringify(withVersion, undefined, 2));
    }
    static loadManifest(filePath, schema, preprocess) {
        let obj = JSON.parse(fs.readFileSync(filePath, { encoding: 'utf-8' }));
        if (preprocess) {
            obj = preprocess(obj);
        }
        Manifest.validate(obj, schema);
        return obj;
    }
    /**
     * This requires some explaining...
     *
     * We previously used `{ Key, Value }` for the object that represents a stack tag. (Notice the casing)
     * @link https://github.com/aws/aws-cdk/blob/v1.27.0/packages/aws-cdk/lib/api/cxapp/stacks.ts#L427.
     *
     * When that object moved to this package, it had to be JSII compliant, which meant the property
     * names must be `camelCased`, and not `PascalCased`. This meant it no longer matches the structure in the `manifest.json` file.
     * In order to support current manifest files, we have to translate the `PascalCased` representation to the new `camelCased` one.
     *
     * Note that the serialization itself still writes `PascalCased` because it relates to how CloudFormation expects it.
     *
     * Ideally, we would start writing the `camelCased` and translate to how CloudFormation expects it when needed. But this requires nasty
     * backwards-compatibility code and it just doesn't seem to be worth the effort.
     */
    static patchStackTagsOnRead(manifest) {
        return Manifest.replaceStackTags(manifest, tags => tags.map((diskTag) => ({
            key: diskTag.Key,
            value: diskTag.Value,
        })));
    }
    /**
     * See explanation on `patchStackTagsOnRead`
     *
     * Translate stack tags metadata if it has the "right" casing.
     */
    static patchStackTagsOnWrite(manifest) {
        return Manifest.replaceStackTags(manifest, tags => tags.map(memTag => 
        // Might already be uppercased (because stack synthesis generates it in final form yet)
        ('Key' in memTag ? memTag : { Key: memTag.key, Value: memTag.value })));
    }
    /**
     * Recursively replace stack tags in the stack metadata
     */
    static replaceStackTags(manifest, fn) {
        // Need to add in the `noUndefined`s because otherwise jest snapshot tests are going to freak out
        // about the keys with values that are `undefined` (even though they would never be JSON.stringified)
        return noUndefined({
            ...manifest,
            artifacts: mapValues(manifest.artifacts, artifact => {
                if (artifact.type !== assembly.ArtifactType.AWS_CLOUDFORMATION_STACK) {
                    return artifact;
                }
                return noUndefined({
                    ...artifact,
                    metadata: mapValues(artifact.metadata, metadataEntries => metadataEntries.map(metadataEntry => {
                        if (metadataEntry.type !== assembly.ArtifactMetadataEntryType.STACK_TAGS || !metadataEntry.data) {
                            return metadataEntry;
                        }
                        return {
                            ...metadataEntry,
                            data: fn(metadataEntry.data),
                        };
                    })),
                });
            }),
        });
    }
    constructor() { }
}
exports.Manifest = Manifest;
function mapValues(xs, fn) {
    if (!xs) {
        return undefined;
    }
    const ret = {};
    for (const [k, v] of Object.entries(xs)) {
        ret[k] = fn(v);
    }
    return ret;
}
function noUndefined(xs) {
    const ret = {};
    for (const [k, v] of Object.entries(xs)) {
        if (v !== undefined) {
            ret[k] = v;
        }
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibWFuaWZlc3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJtYW5pZmVzdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSx5QkFBeUI7QUFDekIseUNBQXlDO0FBQ3pDLGlDQUFpQztBQUVqQyw2Q0FBNkM7QUFDN0MsdURBQXVEO0FBQ3ZELDBEQUEwRDtBQUMxRCxrRUFBa0U7QUFDbEUsaUVBQWlFO0FBQ2pFLDZCQUE2QjtBQUNoQixRQUFBLGdCQUFnQixHQUFXLHdDQUF3QyxDQUFDO0FBQ2pGLE1BQU0sYUFBYSxHQUFHLE9BQU8sQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO0FBQzlELE1BQU0sZUFBZSxHQUFHLE9BQU8sQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDO0FBQ3hFOztHQUVHO0FBQ0gsTUFBTSxjQUFjLEdBQUcsT0FBTyxDQUFDLHVDQUF1QyxDQUFDLENBQUMsT0FBTyxDQUFDO0FBQ2hGOztHQUVHO0FBQ0gsTUFBYSxRQUFRO0lBQ2pCOzs7OztPQUtHO0lBQ0ksTUFBTSxDQUFDLG9CQUFvQixDQUFDLFFBQW1DLEVBQUUsUUFBZ0I7UUFDcEYsUUFBUSxDQUFDLFlBQVksQ0FBQyxRQUFRLEVBQUUsUUFBUSxFQUFFLGVBQWUsRUFBRSxRQUFRLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUMvRixDQUFDO0lBQ0Q7Ozs7T0FJRztJQUNJLE1BQU0sQ0FBQyxvQkFBb0IsQ0FBQyxRQUFnQjtRQUMvQyxPQUFPLFFBQVEsQ0FBQyxZQUFZLENBQUMsUUFBUSxFQUFFLGVBQWUsRUFBRSxRQUFRLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUMzRixDQUFDO0lBQ0Q7Ozs7O09BS0c7SUFDSSxNQUFNLENBQUMsaUJBQWlCLENBQUMsUUFBOEIsRUFBRSxRQUFnQjtRQUM1RSxRQUFRLENBQUMsWUFBWSxDQUFDLFFBQVEsRUFBRSxRQUFRLEVBQUUsYUFBYSxFQUFFLFFBQVEsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0lBQzVGLENBQUM7SUFDRDs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLGlCQUFpQixDQUFDLFFBQWdCO1FBQzVDLE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQyxRQUFRLEVBQUUsYUFBYSxDQUFDLENBQUM7SUFDdEQsQ0FBQztJQUNEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLE9BQU87UUFDakIsT0FBTyxjQUFjLENBQUM7SUFDMUIsQ0FBQztJQUNEOzs7T0FHRztJQUNJLE1BQU0sQ0FBQyxJQUFJLENBQUMsUUFBbUMsRUFBRSxRQUFnQixJQUFJLE9BQU8sSUFBSSxDQUFDLG9CQUFvQixDQUFDLFFBQVEsRUFBRSxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDbkk7OztPQUdHO0lBQ0ksTUFBTSxDQUFDLElBQUksQ0FBQyxRQUFnQixJQUErQixPQUFPLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDdkcsTUFBTSxDQUFDLFFBQVEsQ0FBQyxRQUV2QixFQUFFLE1BQXlCO1FBQ3hCLFNBQVMsWUFBWSxDQUFDLE9BQWU7WUFDakMsTUFBTSxHQUFHLEdBQUcsTUFBTSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztZQUNsQyxJQUFJLENBQUMsR0FBRyxFQUFFO2dCQUNOLE1BQU0sSUFBSSxLQUFLLENBQUMsMkJBQTJCLE9BQU8sR0FBRyxDQUFDLENBQUM7YUFDMUQ7WUFDRCxPQUFPLEdBQUcsQ0FBQztRQUNmLENBQUM7UUFDRCxNQUFNLFlBQVksR0FBRyxZQUFZLENBQUMsUUFBUSxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUM7UUFDdEQsTUFBTSxNQUFNLEdBQUcsWUFBWSxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUM5QyxpREFBaUQ7UUFDakQsSUFBSSxNQUFNLENBQUMsRUFBRSxDQUFDLE1BQU0sRUFBRSxZQUFZLENBQUMsRUFBRTtZQUNqQyxvRkFBb0Y7WUFDcEYsMkNBQTJDO1lBQzNDLE1BQU0sSUFBSSxLQUFLLENBQUMsR0FBRyx3QkFBZ0IseUNBQXlDLFlBQVksZUFBZSxNQUFNLEVBQUUsQ0FBQyxDQUFDO1NBQ3BIO1FBQ0QsbUNBQW1DO1FBQ25DLE1BQU0sU0FBUyxHQUFHLElBQUksVUFBVSxDQUFDLFNBQVMsRUFBRSxDQUFDO1FBQzdDLE1BQU0sTUFBTSxHQUFHLFNBQVMsQ0FBQyxRQUFRLENBQUMsUUFBUSxFQUFFLE1BQU0sRUFBRTtZQUNoRCxzREFBc0Q7WUFDdEQsWUFBWSxFQUFFLElBQUk7WUFDbEIsc0JBQXNCLEVBQUUsS0FBSztTQUN6QixDQUFDLENBQUM7UUFDVixJQUFJLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRTtZQUNmLE1BQU0sSUFBSSxLQUFLLENBQUMsK0JBQStCLE1BQU0sRUFBRSxDQUFDLENBQUM7U0FDNUQ7SUFDTCxDQUFDO0lBQ08sTUFBTSxDQUFDLFlBQVksQ0FBQyxRQUFhLEVBQUUsUUFBZ0IsRUFBRSxNQUF5QixFQUFFLFVBQThCO1FBQ2xILElBQUksV0FBVyxHQUFHLEVBQUUsR0FBRyxRQUFRLEVBQUUsT0FBTyxFQUFFLFFBQVEsQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDO1FBQy9ELFFBQVEsQ0FBQyxRQUFRLENBQUMsV0FBVyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQ3ZDLElBQUksVUFBVSxFQUFFO1lBQ1osV0FBVyxHQUFHLFVBQVUsQ0FBQyxXQUFXLENBQUMsQ0FBQztTQUN6QztRQUNELEVBQUUsQ0FBQyxhQUFhLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsV0FBVyxFQUFFLFNBQVMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQzFFLENBQUM7SUFDTyxNQUFNLENBQUMsWUFBWSxDQUFDLFFBQWdCLEVBQUUsTUFBeUIsRUFBRSxVQUE4QjtRQUNuRyxJQUFJLEdBQUcsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxZQUFZLENBQUMsUUFBUSxFQUFFLEVBQUUsUUFBUSxFQUFFLE9BQU8sRUFBRSxDQUFDLENBQUMsQ0FBQztRQUN2RSxJQUFJLFVBQVUsRUFBRTtZQUNaLEdBQUcsR0FBRyxVQUFVLENBQUMsR0FBRyxDQUFDLENBQUM7U0FDekI7UUFDRCxRQUFRLENBQUMsUUFBUSxDQUFDLEdBQUcsRUFBRSxNQUFNLENBQUMsQ0FBQztRQUMvQixPQUFPLEdBQUcsQ0FBQztJQUNmLENBQUM7SUFDRDs7Ozs7Ozs7Ozs7Ozs7T0FjRztJQUNLLE1BQU0sQ0FBQyxvQkFBb0IsQ0FBQyxRQUFtQztRQUNuRSxPQUFPLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQyxRQUFRLEVBQUUsSUFBSSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsT0FBWSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1lBQzNFLEdBQUcsRUFBRSxPQUFPLENBQUMsR0FBRztZQUNoQixLQUFLLEVBQUUsT0FBTyxDQUFDLEtBQUs7U0FDdkIsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUNULENBQUM7SUFDRDs7OztPQUlHO0lBQ0ssTUFBTSxDQUFDLHFCQUFxQixDQUFDLFFBQW1DO1FBQ3BFLE9BQU8sUUFBUSxDQUFDLGdCQUFnQixDQUFDLFFBQVEsRUFBRSxJQUFJLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLEVBQUU7UUFDckUsdUZBQXVGO1FBQ3ZGLENBQUMsS0FBSyxJQUFJLE1BQU0sQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLEdBQUcsRUFBRSxNQUFNLENBQUMsR0FBRyxFQUFFLEtBQUssRUFBRSxNQUFNLENBQUMsS0FBSyxFQUFFLENBQVEsQ0FBQyxDQUFDLENBQUM7SUFDbkYsQ0FBQztJQUNEOztPQUVHO0lBQ0ssTUFBTSxDQUFDLGdCQUFnQixDQUFDLFFBQW1DLEVBQUUsRUFBZ0Q7UUFDakgsaUdBQWlHO1FBQ2pHLHFHQUFxRztRQUNyRyxPQUFPLFdBQVcsQ0FBQztZQUNmLEdBQUcsUUFBUTtZQUNYLFNBQVMsRUFBRSxTQUFTLENBQUMsUUFBUSxDQUFDLFNBQVMsRUFBRSxRQUFRLENBQUMsRUFBRTtnQkFDaEQsSUFBSSxRQUFRLENBQUMsSUFBSSxLQUFLLFFBQVEsQ0FBQyxZQUFZLENBQUMsd0JBQXdCLEVBQUU7b0JBQ2xFLE9BQU8sUUFBUSxDQUFDO2lCQUNuQjtnQkFDRCxPQUFPLFdBQVcsQ0FBQztvQkFDZixHQUFHLFFBQVE7b0JBQ1gsUUFBUSxFQUFFLFNBQVMsQ0FBQyxRQUFRLENBQUMsUUFBUSxFQUFFLGVBQWUsQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsRUFBRTt3QkFDMUYsSUFBSSxhQUFhLENBQUMsSUFBSSxLQUFLLFFBQVEsQ0FBQyx5QkFBeUIsQ0FBQyxVQUFVLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFOzRCQUM3RixPQUFPLGFBQWEsQ0FBQzt5QkFDeEI7d0JBQ0QsT0FBTzs0QkFDSCxHQUFHLGFBQWE7NEJBQ2hCLElBQUksRUFBRSxFQUFFLENBQUMsYUFBYSxDQUFDLElBQXVDLENBQUM7eUJBQ2xFLENBQUM7b0JBQ04sQ0FBQyxDQUFDLENBQUM7aUJBQ3VCLENBQUMsQ0FBQztZQUNwQyxDQUFDLENBQUM7U0FDTCxDQUFDLENBQUM7SUFDUCxDQUFDO0lBQ0QsZ0JBQXdCLENBQUM7Q0FDNUI7QUEzSkQsNEJBMkpDO0FBRUQsU0FBUyxTQUFTLENBQU8sRUFBaUMsRUFBRSxFQUFlO0lBQ3ZFLElBQUksQ0FBQyxFQUFFLEVBQUU7UUFDTCxPQUFPLFNBQVMsQ0FBQztLQUNwQjtJQUNELE1BQU0sR0FBRyxHQUFrQyxFQUFFLENBQUM7SUFDOUMsS0FBSyxNQUFNLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLEVBQUU7UUFDckMsR0FBRyxDQUFDLENBQUMsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztLQUNsQjtJQUNELE9BQU8sR0FBRyxDQUFDO0FBQ2YsQ0FBQztBQUNELFNBQVMsV0FBVyxDQUFtQixFQUFLO0lBQ3hDLE1BQU0sR0FBRyxHQUFRLEVBQUUsQ0FBQztJQUNwQixLQUFLLE1BQU0sQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsRUFBRTtRQUNyQyxJQUFJLENBQUMsS0FBSyxTQUFTLEVBQUU7WUFDakIsR0FBRyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQztTQUNkO0tBQ0o7SUFDRCxPQUFPLEdBQUcsQ0FBQztBQUNmLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBmcyBmcm9tICdmcyc7XG5pbXBvcnQgKiBhcyBqc29uc2NoZW1hIGZyb20gJ2pzb25zY2hlbWEnO1xuaW1wb3J0ICogYXMgc2VtdmVyIGZyb20gJ3NlbXZlcic7XG5pbXBvcnQgKiBhcyBhc3NldHMgZnJvbSAnLi9hc3NldHMnO1xuaW1wb3J0ICogYXMgYXNzZW1ibHkgZnJvbSAnLi9jbG91ZC1hc3NlbWJseSc7XG4vKiBlc2xpbnQtZGlzYWJsZSBAdHlwZXNjcmlwdC1lc2xpbnQvbm8tdmFyLXJlcXVpcmVzICovXG4vKiBlc2xpbnQtZGlzYWJsZSBAdHlwZXNjcmlwdC1lc2xpbnQvbm8tcmVxdWlyZS1pbXBvcnRzICovXG4vLyB0aGlzIHByZWZpeCBpcyB1c2VkIGJ5IHRoZSBDTEkgdG8gaWRlbnRpZnkgdGhpcyBzcGVjaWZpYyBlcnJvci5cbi8vIGluIHdoaWNoIGNhc2Ugd2Ugd2FudCB0byBpbnN0cnVjdCB0aGUgdXNlciB0byB1cGdyYWRlIGhpcyBDTEkuXG4vLyBzZWUgZXhlYy50cyNjcmVhdGVBc3NlbWJseVxuZXhwb3J0IGNvbnN0IFZFUlNJT05fTUlTTUFUQ0g6IHN0cmluZyA9ICdDbG91ZCBhc3NlbWJseSBzY2hlbWEgdmVyc2lvbiBtaXNtYXRjaCc7XG5jb25zdCBBU1NFVFNfU0NIRU1BID0gcmVxdWlyZSgnLi4vc2NoZW1hL2Fzc2V0cy5zY2hlbWEuanNvbicpO1xuY29uc3QgQVNTRU1CTFlfU0NIRU1BID0gcmVxdWlyZSgnLi4vc2NoZW1hL2Nsb3VkLWFzc2VtYmx5LnNjaGVtYS5qc29uJyk7XG4vKipcbiAqIFZlcnNpb24gaXMgc2hhcmVkIGZvciBib3RoIG1hbmlmZXN0c1xuICovXG5jb25zdCBTQ0hFTUFfVkVSU0lPTiA9IHJlcXVpcmUoJy4uL3NjaGVtYS9jbG91ZC1hc3NlbWJseS52ZXJzaW9uLmpzb24nKS52ZXJzaW9uO1xuLyoqXG4gKiBQcm90b2NvbCB1dGlsaXR5IGNsYXNzLlxuICovXG5leHBvcnQgY2xhc3MgTWFuaWZlc3Qge1xuICAgIC8qKlxuICAgICAqIFZhbGlkYXRlcyBhbmQgc2F2ZXMgdGhlIGNsb3VkIGFzc2VtYmx5IG1hbmlmZXN0IHRvIGZpbGUuXG4gICAgICpcbiAgICAgKiBAcGFyYW0gbWFuaWZlc3QgLSBtYW5pZmVzdC5cbiAgICAgKiBAcGFyYW0gZmlsZVBhdGggLSBvdXRwdXQgZmlsZSBwYXRoLlxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgc2F2ZUFzc2VtYmx5TWFuaWZlc3QobWFuaWZlc3Q6IGFzc2VtYmx5LkFzc2VtYmx5TWFuaWZlc3QsIGZpbGVQYXRoOiBzdHJpbmcpIHtcbiAgICAgICAgTWFuaWZlc3Quc2F2ZU1hbmlmZXN0KG1hbmlmZXN0LCBmaWxlUGF0aCwgQVNTRU1CTFlfU0NIRU1BLCBNYW5pZmVzdC5wYXRjaFN0YWNrVGFnc09uV3JpdGUpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBMb2FkIGFuZCB2YWxpZGF0ZXMgdGhlIGNsb3VkIGFzc2VtYmx5IG1hbmlmZXN0IGZyb20gZmlsZS5cbiAgICAgKlxuICAgICAqIEBwYXJhbSBmaWxlUGF0aCAtIHBhdGggdG8gdGhlIG1hbmlmZXN0IGZpbGUuXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBsb2FkQXNzZW1ibHlNYW5pZmVzdChmaWxlUGF0aDogc3RyaW5nKTogYXNzZW1ibHkuQXNzZW1ibHlNYW5pZmVzdCB7XG4gICAgICAgIHJldHVybiBNYW5pZmVzdC5sb2FkTWFuaWZlc3QoZmlsZVBhdGgsIEFTU0VNQkxZX1NDSEVNQSwgTWFuaWZlc3QucGF0Y2hTdGFja1RhZ3NPblJlYWQpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBWYWxpZGF0ZXMgYW5kIHNhdmVzIHRoZSBhc3NldCBtYW5pZmVzdCB0byBmaWxlLlxuICAgICAqXG4gICAgICogQHBhcmFtIG1hbmlmZXN0IC0gbWFuaWZlc3QuXG4gICAgICogQHBhcmFtIGZpbGVQYXRoIC0gb3V0cHV0IGZpbGUgcGF0aC5cbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIHNhdmVBc3NldE1hbmlmZXN0KG1hbmlmZXN0OiBhc3NldHMuQXNzZXRNYW5pZmVzdCwgZmlsZVBhdGg6IHN0cmluZykge1xuICAgICAgICBNYW5pZmVzdC5zYXZlTWFuaWZlc3QobWFuaWZlc3QsIGZpbGVQYXRoLCBBU1NFVFNfU0NIRU1BLCBNYW5pZmVzdC5wYXRjaFN0YWNrVGFnc09uUmVhZCk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIExvYWQgYW5kIHZhbGlkYXRlcyB0aGUgYXNzZXQgbWFuaWZlc3QgZnJvbSBmaWxlLlxuICAgICAqXG4gICAgICogQHBhcmFtIGZpbGVQYXRoIC0gcGF0aCB0byB0aGUgbWFuaWZlc3QgZmlsZS5cbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGxvYWRBc3NldE1hbmlmZXN0KGZpbGVQYXRoOiBzdHJpbmcpOiBhc3NldHMuQXNzZXRNYW5pZmVzdCB7XG4gICAgICAgIHJldHVybiB0aGlzLmxvYWRNYW5pZmVzdChmaWxlUGF0aCwgQVNTRVRTX1NDSEVNQSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEZldGNoIHRoZSBjdXJyZW50IHNjaGVtYSB2ZXJzaW9uIG51bWJlci5cbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIHZlcnNpb24oKTogc3RyaW5nIHtcbiAgICAgICAgcmV0dXJuIFNDSEVNQV9WRVJTSU9OO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBEZXByZWNhdGVkXG4gICAgICogQGRlcHJlY2F0ZWQgdXNlIGBzYXZlQXNzZW1ibHlNYW5pZmVzdCgpYFxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgc2F2ZShtYW5pZmVzdDogYXNzZW1ibHkuQXNzZW1ibHlNYW5pZmVzdCwgZmlsZVBhdGg6IHN0cmluZykgeyByZXR1cm4gdGhpcy5zYXZlQXNzZW1ibHlNYW5pZmVzdChtYW5pZmVzdCwgZmlsZVBhdGgpOyB9XG4gICAgLyoqXG4gICAgICogRGVwcmVjYXRlZFxuICAgICAqIEBkZXByZWNhdGVkIHVzZSBgbG9hZEFzc2VtYmx5TWFuaWZlc3QoKWBcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGxvYWQoZmlsZVBhdGg6IHN0cmluZyk6IGFzc2VtYmx5LkFzc2VtYmx5TWFuaWZlc3QgeyByZXR1cm4gdGhpcy5sb2FkQXNzZW1ibHlNYW5pZmVzdChmaWxlUGF0aCk7IH1cbiAgICBwcml2YXRlIHN0YXRpYyB2YWxpZGF0ZShtYW5pZmVzdDoge1xuICAgICAgICB2ZXJzaW9uOiBzdHJpbmc7XG4gICAgfSwgc2NoZW1hOiBqc29uc2NoZW1hLlNjaGVtYSkge1xuICAgICAgICBmdW5jdGlvbiBwYXJzZVZlcnNpb24odmVyc2lvbjogc3RyaW5nKSB7XG4gICAgICAgICAgICBjb25zdCB2ZXIgPSBzZW12ZXIudmFsaWQodmVyc2lvbik7XG4gICAgICAgICAgICBpZiAoIXZlcikge1xuICAgICAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgSW52YWxpZCBzZW12ZXIgc3RyaW5nOiBcIiR7dmVyc2lvbn1cImApO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgcmV0dXJuIHZlcjtcbiAgICAgICAgfVxuICAgICAgICBjb25zdCBtYXhTdXBwb3J0ZWQgPSBwYXJzZVZlcnNpb24oTWFuaWZlc3QudmVyc2lvbigpKTtcbiAgICAgICAgY29uc3QgYWN0dWFsID0gcGFyc2VWZXJzaW9uKG1hbmlmZXN0LnZlcnNpb24pO1xuICAgICAgICAvLyBmaXJzdCB2YWxpZGF0ZSB0aGUgdmVyc2lvbiBzaG91bGQgYmUgYWNjZXB0ZWQuXG4gICAgICAgIGlmIChzZW12ZXIuZ3QoYWN0dWFsLCBtYXhTdXBwb3J0ZWQpKSB7XG4gICAgICAgICAgICAvLyB3ZSB1c2UgYSB3ZWxsIGtub3duIGVycm9yIHByZWZpeCBzbyB0aGF0IHRoZSBDTEkgY2FuIGlkZW50aWZ5IHRoaXMgc3BlY2lmaWMgZXJyb3JcbiAgICAgICAgICAgIC8vIGFuZCBwcmludCBzb21lIG1vcmUgY29udGV4dCB0byB0aGUgdXNlci5cbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgJHtWRVJTSU9OX01JU01BVENIfTogTWF4aW11bSBzY2hlbWEgdmVyc2lvbiBzdXBwb3J0ZWQgaXMgJHttYXhTdXBwb3J0ZWR9LCBidXQgZm91bmQgJHthY3R1YWx9YCk7XG4gICAgICAgIH1cbiAgICAgICAgLy8gbm93IHZhbGlkYXRlIHRoZSBmb3JtYXQgaXMgZ29vZC5cbiAgICAgICAgY29uc3QgdmFsaWRhdG9yID0gbmV3IGpzb25zY2hlbWEuVmFsaWRhdG9yKCk7XG4gICAgICAgIGNvbnN0IHJlc3VsdCA9IHZhbGlkYXRvci52YWxpZGF0ZShtYW5pZmVzdCwgc2NoZW1hLCB7XG4gICAgICAgICAgICAvLyBkb2VzIGV4aXN0IGJ1dCBpcyBub3QgaW4gdGhlIFR5cGVTY3JpcHQgZGVmaW5pdGlvbnNcbiAgICAgICAgICAgIG5lc3RlZEVycm9yczogdHJ1ZSxcbiAgICAgICAgICAgIGFsbG93VW5rbm93bkF0dHJpYnV0ZXM6IGZhbHNlLFxuICAgICAgICB9IGFzIGFueSk7XG4gICAgICAgIGlmICghcmVzdWx0LnZhbGlkKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYEludmFsaWQgYXNzZW1ibHkgbWFuaWZlc3Q6XFxuJHtyZXN1bHR9YCk7XG4gICAgICAgIH1cbiAgICB9XG4gICAgcHJpdmF0ZSBzdGF0aWMgc2F2ZU1hbmlmZXN0KG1hbmlmZXN0OiBhbnksIGZpbGVQYXRoOiBzdHJpbmcsIHNjaGVtYToganNvbnNjaGVtYS5TY2hlbWEsIHByZXByb2Nlc3M/OiAob2JqOiBhbnkpID0+IGFueSkge1xuICAgICAgICBsZXQgd2l0aFZlcnNpb24gPSB7IC4uLm1hbmlmZXN0LCB2ZXJzaW9uOiBNYW5pZmVzdC52ZXJzaW9uKCkgfTtcbiAgICAgICAgTWFuaWZlc3QudmFsaWRhdGUod2l0aFZlcnNpb24sIHNjaGVtYSk7XG4gICAgICAgIGlmIChwcmVwcm9jZXNzKSB7XG4gICAgICAgICAgICB3aXRoVmVyc2lvbiA9IHByZXByb2Nlc3Mod2l0aFZlcnNpb24pO1xuICAgICAgICB9XG4gICAgICAgIGZzLndyaXRlRmlsZVN5bmMoZmlsZVBhdGgsIEpTT04uc3RyaW5naWZ5KHdpdGhWZXJzaW9uLCB1bmRlZmluZWQsIDIpKTtcbiAgICB9XG4gICAgcHJpdmF0ZSBzdGF0aWMgbG9hZE1hbmlmZXN0KGZpbGVQYXRoOiBzdHJpbmcsIHNjaGVtYToganNvbnNjaGVtYS5TY2hlbWEsIHByZXByb2Nlc3M/OiAob2JqOiBhbnkpID0+IGFueSkge1xuICAgICAgICBsZXQgb2JqID0gSlNPTi5wYXJzZShmcy5yZWFkRmlsZVN5bmMoZmlsZVBhdGgsIHsgZW5jb2Rpbmc6ICd1dGYtOCcgfSkpO1xuICAgICAgICBpZiAocHJlcHJvY2Vzcykge1xuICAgICAgICAgICAgb2JqID0gcHJlcHJvY2VzcyhvYmopO1xuICAgICAgICB9XG4gICAgICAgIE1hbmlmZXN0LnZhbGlkYXRlKG9iaiwgc2NoZW1hKTtcbiAgICAgICAgcmV0dXJuIG9iajtcbiAgICB9XG4gICAgLyoqXG4gICAgICogVGhpcyByZXF1aXJlcyBzb21lIGV4cGxhaW5pbmcuLi5cbiAgICAgKlxuICAgICAqIFdlIHByZXZpb3VzbHkgdXNlZCBgeyBLZXksIFZhbHVlIH1gIGZvciB0aGUgb2JqZWN0IHRoYXQgcmVwcmVzZW50cyBhIHN0YWNrIHRhZy4gKE5vdGljZSB0aGUgY2FzaW5nKVxuICAgICAqIEBsaW5rIGh0dHBzOi8vZ2l0aHViLmNvbS9hd3MvYXdzLWNkay9ibG9iL3YxLjI3LjAvcGFja2FnZXMvYXdzLWNkay9saWIvYXBpL2N4YXBwL3N0YWNrcy50cyNMNDI3LlxuICAgICAqXG4gICAgICogV2hlbiB0aGF0IG9iamVjdCBtb3ZlZCB0byB0aGlzIHBhY2thZ2UsIGl0IGhhZCB0byBiZSBKU0lJIGNvbXBsaWFudCwgd2hpY2ggbWVhbnQgdGhlIHByb3BlcnR5XG4gICAgICogbmFtZXMgbXVzdCBiZSBgY2FtZWxDYXNlZGAsIGFuZCBub3QgYFBhc2NhbENhc2VkYC4gVGhpcyBtZWFudCBpdCBubyBsb25nZXIgbWF0Y2hlcyB0aGUgc3RydWN0dXJlIGluIHRoZSBgbWFuaWZlc3QuanNvbmAgZmlsZS5cbiAgICAgKiBJbiBvcmRlciB0byBzdXBwb3J0IGN1cnJlbnQgbWFuaWZlc3QgZmlsZXMsIHdlIGhhdmUgdG8gdHJhbnNsYXRlIHRoZSBgUGFzY2FsQ2FzZWRgIHJlcHJlc2VudGF0aW9uIHRvIHRoZSBuZXcgYGNhbWVsQ2FzZWRgIG9uZS5cbiAgICAgKlxuICAgICAqIE5vdGUgdGhhdCB0aGUgc2VyaWFsaXphdGlvbiBpdHNlbGYgc3RpbGwgd3JpdGVzIGBQYXNjYWxDYXNlZGAgYmVjYXVzZSBpdCByZWxhdGVzIHRvIGhvdyBDbG91ZEZvcm1hdGlvbiBleHBlY3RzIGl0LlxuICAgICAqXG4gICAgICogSWRlYWxseSwgd2Ugd291bGQgc3RhcnQgd3JpdGluZyB0aGUgYGNhbWVsQ2FzZWRgIGFuZCB0cmFuc2xhdGUgdG8gaG93IENsb3VkRm9ybWF0aW9uIGV4cGVjdHMgaXQgd2hlbiBuZWVkZWQuIEJ1dCB0aGlzIHJlcXVpcmVzIG5hc3R5XG4gICAgICogYmFja3dhcmRzLWNvbXBhdGliaWxpdHkgY29kZSBhbmQgaXQganVzdCBkb2Vzbid0IHNlZW0gdG8gYmUgd29ydGggdGhlIGVmZm9ydC5cbiAgICAgKi9cbiAgICBwcml2YXRlIHN0YXRpYyBwYXRjaFN0YWNrVGFnc09uUmVhZChtYW5pZmVzdDogYXNzZW1ibHkuQXNzZW1ibHlNYW5pZmVzdCkge1xuICAgICAgICByZXR1cm4gTWFuaWZlc3QucmVwbGFjZVN0YWNrVGFncyhtYW5pZmVzdCwgdGFncyA9PiB0YWdzLm1hcCgoZGlza1RhZzogYW55KSA9PiAoe1xuICAgICAgICAgICAga2V5OiBkaXNrVGFnLktleSxcbiAgICAgICAgICAgIHZhbHVlOiBkaXNrVGFnLlZhbHVlLFxuICAgICAgICB9KSkpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBTZWUgZXhwbGFuYXRpb24gb24gYHBhdGNoU3RhY2tUYWdzT25SZWFkYFxuICAgICAqXG4gICAgICogVHJhbnNsYXRlIHN0YWNrIHRhZ3MgbWV0YWRhdGEgaWYgaXQgaGFzIHRoZSBcInJpZ2h0XCIgY2FzaW5nLlxuICAgICAqL1xuICAgIHByaXZhdGUgc3RhdGljIHBhdGNoU3RhY2tUYWdzT25Xcml0ZShtYW5pZmVzdDogYXNzZW1ibHkuQXNzZW1ibHlNYW5pZmVzdCkge1xuICAgICAgICByZXR1cm4gTWFuaWZlc3QucmVwbGFjZVN0YWNrVGFncyhtYW5pZmVzdCwgdGFncyA9PiB0YWdzLm1hcChtZW1UYWcgPT4gXG4gICAgICAgIC8vIE1pZ2h0IGFscmVhZHkgYmUgdXBwZXJjYXNlZCAoYmVjYXVzZSBzdGFjayBzeW50aGVzaXMgZ2VuZXJhdGVzIGl0IGluIGZpbmFsIGZvcm0geWV0KVxuICAgICAgICAoJ0tleScgaW4gbWVtVGFnID8gbWVtVGFnIDogeyBLZXk6IG1lbVRhZy5rZXksIFZhbHVlOiBtZW1UYWcudmFsdWUgfSkgYXMgYW55KSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFJlY3Vyc2l2ZWx5IHJlcGxhY2Ugc3RhY2sgdGFncyBpbiB0aGUgc3RhY2sgbWV0YWRhdGFcbiAgICAgKi9cbiAgICBwcml2YXRlIHN0YXRpYyByZXBsYWNlU3RhY2tUYWdzKG1hbmlmZXN0OiBhc3NlbWJseS5Bc3NlbWJseU1hbmlmZXN0LCBmbjogRW5kb2Z1bmN0b3I8YXNzZW1ibHkuU3RhY2tUYWdzTWV0YWRhdGFFbnRyeT4pOiBhc3NlbWJseS5Bc3NlbWJseU1hbmlmZXN0IHtcbiAgICAgICAgLy8gTmVlZCB0byBhZGQgaW4gdGhlIGBub1VuZGVmaW5lZGBzIGJlY2F1c2Ugb3RoZXJ3aXNlIGplc3Qgc25hcHNob3QgdGVzdHMgYXJlIGdvaW5nIHRvIGZyZWFrIG91dFxuICAgICAgICAvLyBhYm91dCB0aGUga2V5cyB3aXRoIHZhbHVlcyB0aGF0IGFyZSBgdW5kZWZpbmVkYCAoZXZlbiB0aG91Z2ggdGhleSB3b3VsZCBuZXZlciBiZSBKU09OLnN0cmluZ2lmaWVkKVxuICAgICAgICByZXR1cm4gbm9VbmRlZmluZWQoe1xuICAgICAgICAgICAgLi4ubWFuaWZlc3QsXG4gICAgICAgICAgICBhcnRpZmFjdHM6IG1hcFZhbHVlcyhtYW5pZmVzdC5hcnRpZmFjdHMsIGFydGlmYWN0ID0+IHtcbiAgICAgICAgICAgICAgICBpZiAoYXJ0aWZhY3QudHlwZSAhPT0gYXNzZW1ibHkuQXJ0aWZhY3RUeXBlLkFXU19DTE9VREZPUk1BVElPTl9TVEFDSykge1xuICAgICAgICAgICAgICAgICAgICByZXR1cm4gYXJ0aWZhY3Q7XG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgIHJldHVybiBub1VuZGVmaW5lZCh7XG4gICAgICAgICAgICAgICAgICAgIC4uLmFydGlmYWN0LFxuICAgICAgICAgICAgICAgICAgICBtZXRhZGF0YTogbWFwVmFsdWVzKGFydGlmYWN0Lm1ldGFkYXRhLCBtZXRhZGF0YUVudHJpZXMgPT4gbWV0YWRhdGFFbnRyaWVzLm1hcChtZXRhZGF0YUVudHJ5ID0+IHtcbiAgICAgICAgICAgICAgICAgICAgICAgIGlmIChtZXRhZGF0YUVudHJ5LnR5cGUgIT09IGFzc2VtYmx5LkFydGlmYWN0TWV0YWRhdGFFbnRyeVR5cGUuU1RBQ0tfVEFHUyB8fCAhbWV0YWRhdGFFbnRyeS5kYXRhKSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgcmV0dXJuIG1ldGFkYXRhRW50cnk7XG4gICAgICAgICAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIC4uLm1ldGFkYXRhRW50cnksXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgZGF0YTogZm4obWV0YWRhdGFFbnRyeS5kYXRhIGFzIGFzc2VtYmx5LlN0YWNrVGFnc01ldGFkYXRhRW50cnkpLFxuICAgICAgICAgICAgICAgICAgICAgICAgfTtcbiAgICAgICAgICAgICAgICAgICAgfSkpLFxuICAgICAgICAgICAgICAgIH0gYXMgYXNzZW1ibHkuQXJ0aWZhY3RNYW5pZmVzdCk7XG4gICAgICAgICAgICB9KSxcbiAgICAgICAgfSk7XG4gICAgfVxuICAgIHByaXZhdGUgY29uc3RydWN0b3IoKSB7IH1cbn1cbnR5cGUgRW5kb2Z1bmN0b3I8QT4gPSAoeDogQSkgPT4gQTtcbmZ1bmN0aW9uIG1hcFZhbHVlczxBLCBCPih4czogUmVjb3JkPHN0cmluZywgQT4gfCB1bmRlZmluZWQsIGZuOiAoeDogQSkgPT4gQik6IFJlY29yZDxzdHJpbmcsIEI+IHwgdW5kZWZpbmVkIHtcbiAgICBpZiAoIXhzKSB7XG4gICAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgfVxuICAgIGNvbnN0IHJldDogUmVjb3JkPHN0cmluZywgQj4gfCB1bmRlZmluZWQgPSB7fTtcbiAgICBmb3IgKGNvbnN0IFtrLCB2XSBvZiBPYmplY3QuZW50cmllcyh4cykpIHtcbiAgICAgICAgcmV0W2tdID0gZm4odik7XG4gICAgfVxuICAgIHJldHVybiByZXQ7XG59XG5mdW5jdGlvbiBub1VuZGVmaW5lZDxBIGV4dGVuZHMgb2JqZWN0Pih4czogQSk6IEEge1xuICAgIGNvbnN0IHJldDogYW55ID0ge307XG4gICAgZm9yIChjb25zdCBbaywgdl0gb2YgT2JqZWN0LmVudHJpZXMoeHMpKSB7XG4gICAgICAgIGlmICh2ICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICAgIHJldFtrXSA9IHY7XG4gICAgICAgIH1cbiAgICB9XG4gICAgcmV0dXJuIHJldDtcbn1cbiJdfQ==