"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnInclude = void 0;
const core = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cfn_parse = require("../../core/lib/cfn-parse"); // Automatically re-written from '@aws-cdk/core/lib/cfn-parse'
const cfn_type_to_l1_mapping = require("./cfn-type-to-l1-mapping");
const futils = require("./file-utils");
/**
 * Construct to import an existing CloudFormation template file into a CDK application.
 * All resources defined in the template file can be retrieved by calling the {@link getResource} method.
 * Any modifications made on the returned resource objects will be reflected in the resulting CDK template.
 */
class CfnInclude extends core.CfnElement {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.conditions = {};
        this.resources = {};
        this.parameters = {};
        this.mappings = {};
        this.rules = {};
        this.hooks = {};
        this.outputs = {};
        this.nestedStacks = {};
        this.parametersToReplace = props.parameters || {};
        // read the template into a JS object
        this.template = futils.readYamlSync(props.templateFile);
        this.preserveLogicalIds = (_a = props.preserveLogicalIds) !== null && _a !== void 0 ? _a : true;
        // check if all user specified parameter values exist in the template
        for (const logicalId of Object.keys(this.parametersToReplace)) {
            if (!(logicalId in (this.template.Parameters || {}))) {
                throw new Error(`Parameter with logical ID '${logicalId}' was not found in the template`);
            }
        }
        // instantiate the Mappings
        this.mappingsScope = new core.Construct(this, '$Mappings');
        for (const mappingName of Object.keys(this.template.Mappings || {})) {
            this.createMapping(mappingName);
        }
        // instantiate all parameters
        for (const logicalId of Object.keys(this.template.Parameters || {})) {
            this.createParameter(logicalId);
        }
        // instantiate the conditions
        this.conditionsScope = new core.Construct(this, '$Conditions');
        for (const conditionName of Object.keys(this.template.Conditions || {})) {
            this.getOrCreateCondition(conditionName);
        }
        // instantiate the rules
        this.rulesScope = new core.Construct(this, '$Rules');
        for (const ruleName of Object.keys(this.template.Rules || {})) {
            this.createRule(ruleName);
        }
        this.nestedStacksToInclude = props.loadNestedStacks || {};
        // instantiate all resources as CDK L1 objects
        for (const logicalId of Object.keys(this.template.Resources || {})) {
            this.getOrCreateResource(logicalId);
        }
        // verify that all nestedStacks have been instantiated
        for (const nestedStackId of Object.keys(props.loadNestedStacks || {})) {
            if (!(nestedStackId in this.resources)) {
                throw new Error(`Nested Stack with logical ID '${nestedStackId}' was not found in the template`);
            }
        }
        // instantiate the Hooks
        this.hooksScope = new core.Construct(this, '$Hooks');
        for (const hookName of Object.keys(this.template.Hooks || {})) {
            this.createHook(hookName);
        }
        const outputScope = new core.Construct(this, '$Ouputs');
        for (const logicalId of Object.keys(this.template.Outputs || {})) {
            this.createOutput(logicalId, outputScope);
        }
    }
    /**
     * Returns the low-level CfnResource from the template with the given logical ID.
     * Any modifications performed on that resource will be reflected in the resulting CDK template.
     *
     * The returned object will be of the proper underlying class;
     * you can always cast it to the correct type in your code:
     *
     *     // assume the template contains an AWS::S3::Bucket with logical ID 'Bucket'
     *     const cfnBucket = cfnTemplate.getResource('Bucket') as s3.CfnBucket;
     *     // cfnBucket is of type s3.CfnBucket
     *
     * If the template does not contain a resource with the given logical ID,
     * an exception will be thrown.
     *
     * @param logicalId the logical ID of the resource in the CloudFormation template file
     */
    getResource(logicalId) {
        const ret = this.resources[logicalId];
        if (!ret) {
            throw new Error(`Resource with logical ID '${logicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnCondition object from the 'Conditions'
     * section of the CloudFormation template with the given name.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Condition with the given name is not present in the template,
     * throws an exception.
     *
     * @param conditionName the name of the Condition in the CloudFormation template file
     */
    getCondition(conditionName) {
        const ret = this.conditions[conditionName];
        if (!ret) {
            throw new Error(`Condition with name '${conditionName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnParameter object from the 'Parameters'
     * section of the included template.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Parameter with the given name is not present in the template,
     * throws an exception.
     *
     * @param parameterName the name of the parameter to retrieve
     */
    getParameter(parameterName) {
        const ret = this.parameters[parameterName];
        if (!ret) {
            throw new Error(`Parameter with name '${parameterName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnMapping object from the 'Mappings' section of the included template.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Mapping with the given name is not present in the template,
     * an exception will be thrown.
     *
     * @param mappingName the name of the Mapping in the template to retrieve
     */
    getMapping(mappingName) {
        const ret = this.mappings[mappingName];
        if (!ret) {
            throw new Error(`Mapping with name '${mappingName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnOutput object from the 'Outputs'
     * section of the included template.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If an Output with the given name is not present in the template,
     * throws an exception.
     *
     * @param logicalId the name of the output to retrieve
     */
    getOutput(logicalId) {
        const ret = this.outputs[logicalId];
        if (!ret) {
            throw new Error(`Output with logical ID '${logicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnRule object from the 'Rules'
     * section of the CloudFormation template with the given name.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Rule with the given name is not present in the template,
     * an exception will be thrown.
     *
     * @param ruleName the name of the Rule in the CloudFormation template
     */
    getRule(ruleName) {
        const ret = this.rules[ruleName];
        if (!ret) {
            throw new Error(`Rule with name '${ruleName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnHook object from the 'Hooks'
     * section of the included CloudFormation template with the given logical ID.
     * Any modifications performed on the returned object will be reflected in the resulting CDK template.
     *
     * If a Hook with the given logical ID is not present in the template,
     * an exception will be thrown.
     *
     * @param hookLogicalId the logical ID of the Hook in the included CloudFormation template's 'Hooks' section
     */
    getHook(hookLogicalId) {
        const ret = this.hooks[hookLogicalId];
        if (!ret) {
            throw new Error(`Hook with logical ID '${hookLogicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns a loaded NestedStack with name logicalId.
     * For a nested stack to be returned by this method,
     * it must be specified either in the {@link CfnIncludeProps.loadNestedStacks} property,
     * or through the {@link loadNestedStack} method.
     *
     * @param logicalId the ID of the stack to retrieve, as it appears in the template
     */
    getNestedStack(logicalId) {
        if (!this.nestedStacks[logicalId]) {
            if (!this.template.Resources[logicalId]) {
                throw new Error(`Nested Stack with logical ID '${logicalId}' was not found in the template`);
            }
            else if (this.template.Resources[logicalId].Type !== 'AWS::CloudFormation::Stack') {
                throw new Error(`Resource with logical ID '${logicalId}' is not a CloudFormation Stack`);
            }
            else {
                throw new Error(`Nested Stack '${logicalId}' was not included in the parent template. ` +
                    'To retrieve an included nested stack, it must be specified either in the `loadNestedStacks` property, or through the `loadNestedStack` method');
            }
        }
        return this.nestedStacks[logicalId];
    }
    /**
     * Includes a template for a child stack inside of this parent template.
     * A child with this logical ID must exist in the template,
     * and be of type AWS::CloudFormation::Stack.
     * This is equivalent to specifying the value in the {@link CfnIncludeProps.loadNestedStacks}
     * property on object construction.
     *
     * @param logicalId the ID of the stack to retrieve, as it appears in the template
     * @param nestedStackProps the properties of the included child Stack
     * @returns the same {@link IncludedNestedStack} object that {@link getNestedStack} returns for this logical ID
     */
    loadNestedStack(logicalId, nestedStackProps) {
        if (logicalId in this.nestedStacks) {
            throw new Error(`Nested Stack '${logicalId}' was already included in its parent template`);
        }
        const cfnStack = this.resources[logicalId];
        if (!cfnStack) {
            throw new Error(`Nested Stack with logical ID '${logicalId}' was not found in the template`);
        }
        if (cfnStack instanceof core.CfnStack) {
            // delete the old CfnStack child - one will be created by the NestedStack object
            this.node.tryRemoveChild(logicalId);
            // remove the previously created CfnStack resource from the resources map
            delete this.resources[logicalId];
            // createNestedStack() (called by getOrCreateResource()) expects this to be filled
            this.nestedStacksToInclude[logicalId] = nestedStackProps;
            this.getOrCreateResource(logicalId);
            return this.nestedStacks[logicalId];
        }
        else {
            throw new Error(`Nested Stack with logical ID '${logicalId}' is not an AWS::CloudFormation::Stack resource`);
        }
    }
    /** @internal */
    _toCloudFormation() {
        const ret = {};
        for (const section of Object.keys(this.template)) {
            const self = this;
            const finder = {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    var _a;
                    return (_a = self.resources[elementName]) !== null && _a !== void 0 ? _a : self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            };
            const cfnParser = new cfn_parse.CfnParser({
                finder,
                parameters: this.parametersToReplace,
            });
            switch (section) {
                case 'Conditions':
                case 'Mappings':
                case 'Resources':
                case 'Parameters':
                case 'Rules':
                case 'Hooks':
                case 'Outputs':
                    // these are rendered as a side effect of instantiating the L1s
                    break;
                default:
                    ret[section] = cfnParser.parseValue(this.template[section]);
            }
        }
        return ret;
    }
    createMapping(mappingName) {
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findCondition() { throw new Error('Referring to Conditions in Mapping definitions is not allowed'); },
                findMapping() { throw new Error('Referring to other Mappings in Mapping definitions is not allowed'); },
                findRefTarget() { throw new Error('Using Ref expressions in Mapping definitions is not allowed'); },
                findResource() { throw new Error('Using GetAtt expressions in Mapping definitions is not allowed'); },
            },
            parameters: {},
        });
        const cfnMapping = new core.CfnMapping(this.mappingsScope, mappingName, {
            mapping: cfnParser.parseValue(this.template.Mappings[mappingName]),
        });
        this.mappings[mappingName] = cfnMapping;
        this.overrideLogicalIdIfNeeded(cfnMapping, mappingName);
    }
    createParameter(logicalId) {
        if (logicalId in this.parametersToReplace) {
            return;
        }
        const expression = new cfn_parse.CfnParser({
            finder: {
                findResource() { throw new Error('Using GetAtt expressions in Parameter definitions is not allowed'); },
                findRefTarget() { throw new Error('Using Ref expressions in Parameter definitions is not allowed'); },
                findCondition() { throw new Error('Referring to Conditions in Parameter definitions is not allowed'); },
                findMapping() { throw new Error('Referring to Mappings in Parameter definitions is not allowed'); },
            },
            parameters: {},
        }).parseValue(this.template.Parameters[logicalId]);
        const cfnParameter = new core.CfnParameter(this, logicalId, {
            type: expression.Type,
            default: expression.Default,
            allowedPattern: expression.AllowedPattern,
            allowedValues: expression.AllowedValues,
            constraintDescription: expression.ConstraintDescription,
            description: expression.Description,
            maxLength: expression.MaxLength,
            maxValue: expression.MaxValue,
            minLength: expression.MinLength,
            minValue: expression.MinValue,
            noEcho: expression.NoEcho,
        });
        this.overrideLogicalIdIfNeeded(cfnParameter, logicalId);
        this.parameters[logicalId] = cfnParameter;
    }
    createRule(ruleName) {
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findRefTarget(refTarget) {
                    // only parameters can be referenced in Rules
                    return self.parameters[refTarget];
                },
                findResource() { throw new Error('Using GetAtt expressions in Rule definitions is not allowed'); },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
            context: cfn_parse.CfnParsingContext.RULES,
        });
        const ruleProperties = cfnParser.parseValue(this.template.Rules[ruleName]);
        const rule = new core.CfnRule(this.rulesScope, ruleName, {
            ruleCondition: ruleProperties.RuleCondition,
            assertions: ruleProperties.Assertions,
        });
        this.rules[ruleName] = rule;
        this.overrideLogicalIdIfNeeded(rule, ruleName);
    }
    createHook(hookName) {
        var _a;
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    var _a;
                    return (_a = self.resources[elementName]) !== null && _a !== void 0 ? _a : self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
        });
        const hookAttributes = this.template.Hooks[hookName];
        let hook;
        switch (hookAttributes.Type) {
            case 'AWS::CodeDeploy::BlueGreen':
                hook = core.CfnCodeDeployBlueGreenHook._fromCloudFormation(this.hooksScope, hookName, hookAttributes, {
                    parser: cfnParser,
                });
                break;
            default: {
                const hookProperties = (_a = cfnParser.parseValue(hookAttributes.Properties)) !== null && _a !== void 0 ? _a : {};
                hook = new core.CfnHook(this.hooksScope, hookName, {
                    type: hookAttributes.Type,
                    properties: hookProperties,
                });
            }
        }
        this.hooks[hookName] = hook;
        this.overrideLogicalIdIfNeeded(hook, hookName);
    }
    createOutput(logicalId, scope) {
        const self = this;
        const outputAttributes = new cfn_parse.CfnParser({
            finder: {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    var _a;
                    return (_a = self.resources[elementName]) !== null && _a !== void 0 ? _a : self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
        }).parseValue(this.template.Outputs[logicalId]);
        const cfnOutput = new core.CfnOutput(scope, logicalId, {
            value: outputAttributes.Value,
            description: outputAttributes.Description,
            exportName: outputAttributes.Export ? outputAttributes.Export.Name : undefined,
            condition: (() => {
                if (!outputAttributes.Condition) {
                    return undefined;
                }
                else if (this.conditions[outputAttributes.Condition]) {
                    return self.getCondition(outputAttributes.Condition);
                }
                throw new Error(`Output with name '${logicalId}' refers to a Condition with name ` +
                    `'${outputAttributes.Condition}' which was not found in this template`);
            })(),
        });
        this.overrideLogicalIdIfNeeded(cfnOutput, logicalId);
        this.outputs[logicalId] = cfnOutput;
    }
    getOrCreateCondition(conditionName) {
        if (conditionName in this.conditions) {
            return this.conditions[conditionName];
        }
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findResource() { throw new Error('Using GetAtt in Condition definitions is not allowed'); },
                findRefTarget(elementName) {
                    // only Parameters can be referenced in the 'Conditions' section
                    return self.parameters[elementName];
                },
                findCondition(cName) {
                    return cName in (self.template.Conditions || {})
                        ? self.getOrCreateCondition(cName)
                        : undefined;
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            context: cfn_parse.CfnParsingContext.CONDITIONS,
            parameters: this.parametersToReplace,
        });
        const cfnCondition = new core.CfnCondition(this.conditionsScope, conditionName, {
            expression: cfnParser.parseValue(this.template.Conditions[conditionName]),
        });
        this.overrideLogicalIdIfNeeded(cfnCondition, conditionName);
        this.conditions[conditionName] = cfnCondition;
        return cfnCondition;
    }
    getOrCreateResource(logicalId) {
        const ret = this.resources[logicalId];
        if (ret) {
            return ret;
        }
        const resourceAttributes = this.template.Resources[logicalId];
        // fail early for resource attributes we don't support yet
        const knownAttributes = [
            'Condition', 'DependsOn', 'Description', 'Metadata', 'Properties', 'Type', 'Version',
            'CreationPolicy', 'DeletionPolicy', 'UpdatePolicy', 'UpdateReplacePolicy',
        ];
        for (const attribute of Object.keys(resourceAttributes)) {
            if (!knownAttributes.includes(attribute)) {
                throw new Error(`The '${attribute}' resource attribute is not supported by cloudformation-include yet. ` +
                    'Either remove it from the template, or use the CdkInclude class from the core package instead.');
            }
        }
        const self = this;
        const finder = {
            findCondition(conditionName) {
                return self.conditions[conditionName];
            },
            findMapping(mappingName) {
                return self.mappings[mappingName];
            },
            findResource(lId) {
                if (!(lId in (self.template.Resources || {}))) {
                    return undefined;
                }
                return self.getOrCreateResource(lId);
            },
            findRefTarget(elementName) {
                if (elementName in self.parameters) {
                    return self.parameters[elementName];
                }
                return this.findResource(elementName);
            },
        };
        const cfnParser = new cfn_parse.CfnParser({
            finder,
            parameters: this.parametersToReplace,
        });
        let l1Instance;
        if (this.nestedStacksToInclude[logicalId]) {
            l1Instance = this.createNestedStack(logicalId, cfnParser);
        }
        else {
            const l1ClassFqn = cfn_type_to_l1_mapping.lookup(resourceAttributes.Type);
            // The AWS::CloudFormation::CustomResource type corresponds to the CfnCustomResource class.
            // Unfortunately, it's quite useless; it only has a single property, ServiceToken.
            // For that reason, even the CustomResource class from @core doesn't use it!
            // So, special-case the handling of this one resource type
            if (l1ClassFqn && resourceAttributes.Type !== 'AWS::CloudFormation::CustomResource') {
                const options = {
                    parser: cfnParser,
                };
                const [moduleName, ...className] = l1ClassFqn.split('.');
                const module = require(moduleName); // eslint-disable-line @typescript-eslint/no-require-imports
                const jsClassFromModule = module[className.join('.')];
                l1Instance = jsClassFromModule._fromCloudFormation(this, logicalId, resourceAttributes, options);
            }
            else {
                l1Instance = new core.CfnResource(this, logicalId, {
                    type: resourceAttributes.Type,
                    properties: cfnParser.parseValue(resourceAttributes.Properties),
                });
                cfnParser.handleAttributes(l1Instance, resourceAttributes, logicalId);
            }
        }
        this.overrideLogicalIdIfNeeded(l1Instance, logicalId);
        this.resources[logicalId] = l1Instance;
        return l1Instance;
    }
    createNestedStack(nestedStackId, cfnParser) {
        const templateResources = this.template.Resources || {};
        const nestedStackAttributes = templateResources[nestedStackId] || {};
        if (nestedStackAttributes.Type !== 'AWS::CloudFormation::Stack') {
            throw new Error(`Nested Stack with logical ID '${nestedStackId}' is not an AWS::CloudFormation::Stack resource`);
        }
        if (nestedStackAttributes.CreationPolicy) {
            throw new Error('CreationPolicy is not supported by the AWS::CloudFormation::Stack resource');
        }
        if (nestedStackAttributes.UpdatePolicy) {
            throw new Error('UpdatePolicy is not supported by the AWS::CloudFormation::Stack resource');
        }
        const nestedStackProps = cfnParser.parseValue(nestedStackAttributes.Properties);
        const nestedStack = new core.NestedStack(this, nestedStackId, {
            parameters: this.parametersForNestedStack(nestedStackProps.Parameters, nestedStackId),
            notificationArns: nestedStackProps.NotificationArns,
            timeout: nestedStackProps.Timeout,
        });
        const template = new CfnInclude(nestedStack, nestedStackId, this.nestedStacksToInclude[nestedStackId]);
        this.nestedStacks[nestedStackId] = { stack: nestedStack, includedTemplate: template };
        // we know this is never undefined for nested stacks
        const nestedStackResource = nestedStack.nestedStackResource;
        cfnParser.handleAttributes(nestedStackResource, nestedStackAttributes, nestedStackId);
        return nestedStackResource;
    }
    parametersForNestedStack(parameters, nestedStackId) {
        var _a;
        if (parameters == null) {
            return undefined;
        }
        const parametersToReplace = (_a = this.nestedStacksToInclude[nestedStackId].parameters) !== null && _a !== void 0 ? _a : {};
        const ret = {};
        for (const paramName of Object.keys(parameters)) {
            if (!(paramName in parametersToReplace)) {
                ret[paramName] = parameters[paramName];
            }
        }
        return ret;
    }
    overrideLogicalIdIfNeeded(element, id) {
        if (this.preserveLogicalIds) {
            element.overrideLogicalId(id);
        }
    }
}
exports.CfnInclude = CfnInclude;
//# sourceMappingURL=data:application/json;base64,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